package ru.yandex.mail.diffusion.generator;

import lombok.SneakyThrows;
import lombok.Value;
import lombok.val;
import one.util.streamex.StreamEx;
import ru.yandex.mail.diffusion.exception.UnsatisfiedClassRequirementException;

import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Parameter;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;

import static java.util.function.Function.identity;

@Value
class FieldInfo {
    String name;
    String getterName;
    Class type;
    List<Class> genericElements;
}

@Value
class ClassInfo {
    Class type;
    String fieldMatcherQualifiedName;
    String patchApplierQualifiedName;
    List<FieldInfo> fields;

    StreamEx<FieldInfo> fieldsStream() {
        return StreamEx.of(fields);
    }
}

class MetaInfoCollector {
    @SneakyThrows
    private static Class classForName(String name) {
        return Class.forName(name);
    }

    private static FieldInfo extractFieldInfo(PropertyDescriptor descriptor) {
        val getter = descriptor.getReadMethod();
        val type = getter.getReturnType();
        var genericElements = Collections.<Class>emptyList();

        if (type.getTypeParameters().length > 0) {
            val genericType = (ParameterizedType) getter.getGenericReturnType();
            genericElements = StreamEx.of(genericType.getActualTypeArguments())
                .map(Type::getTypeName)
                .map(MetaInfoCollector::classForName)
                .toImmutableList();
        }

        return new FieldInfo(descriptor.getName(), getter.getName(), type, genericElements);
    }

    private static boolean isAllArgsConstructor(Constructor constructor, Set<String> fieldsName) {
        val parametersName = StreamEx.of(constructor.getParameters())
            .map(Parameter::getName)
            .toSet();

        return (fieldsName.size() == parametersName.size()) &&
            fieldsName.containsAll(parametersName);
    }

    private static Optional<Constructor> findAllArgsConstructor(Class type, Collection<FieldInfo> fields) {
        val fieldsName = StreamEx.of(fields)
            .map(FieldInfo::getName)
            .toImmutableSet();

        return StreamEx.of(type.getDeclaredConstructors())
            .findFirst(constructor -> isAllArgsConstructor(constructor, fieldsName));
    }

    @SneakyThrows
    ClassInfo collect(Class<?> type) {
        val canonicalName = type.getCanonicalName();
        if (canonicalName == null) {
            throw new UnsatisfiedClassRequirementException(type, "class should not be either local or anonymous");
        }

        if (type.getSuperclass() != Object.class) {
            throw new UnsatisfiedClassRequirementException(type, "class should not have base class");
        }

        if (type.getTypeParameters().length != 0) {
            throw new UnsatisfiedClassRequirementException(type, "class can't be generic");
        }

        val fieldsInfoMap = StreamEx.of(Introspector.getBeanInfo(type, Object.class).getPropertyDescriptors())
            .map(MetaInfoCollector::extractFieldInfo)
            .toMap(FieldInfo::getName, identity());

        val constructor = findAllArgsConstructor(type, fieldsInfoMap.values())
            .orElseThrow(() ->
                new UnsatisfiedClassRequirementException(
                    type, "Constructor with arguments for all properties not found.\n" +
                          "Be sure that appropriate constructor is declared")
            );

        val fieldsInfo = StreamEx.of(constructor.getParameters())
            .map(Parameter::getName)
            .map(fieldsInfoMap::get)
            .toImmutableList();

        val fieldMatcherQualifiedName = canonicalName + "FieldMatcher";
        val patchApplierQualifiedName = canonicalName + "PatchApplier";
        return new ClassInfo(type, fieldMatcherQualifiedName, patchApplierQualifiedName, fieldsInfo);
    }
}
