package ru.yandex.mail.diffusion.generator;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.CtNewMethod;
import lombok.AllArgsConstructor;
import lombok.SneakyThrows;
import lombok.val;
import one.util.streamex.EntryStream;
import ru.yandex.mail.diffusion.patch.PatchApplier;

import java.util.*;

import static java.util.Map.entry;
import static java.util.stream.Collectors.joining;
import static ru.yandex.mail.diffusion.generator.Utils.format;

@AllArgsConstructor
class PatchApplierGenerator {
    private final ClassPool pool;

    private static boolean isBoxedInteger(Class<?> type) {
        return type == Byte.class
            || type == Short.class
            || type == Integer.class
            || type == Long.class;
    }

    private static String generateApply(FieldInfo info) {
        val type = info.getType();
        val name = info.getName();

        if (type.isPrimitive()) {
            if (type == boolean.class) {
                return "change.getBool()";
            } else if (type == double.class) {
                return "change.getDouble()";
            } else if (type == float.class) {
                return "(float) change.getDouble()";
            } else {
                return format("ApplierOps#${methodPrefix}Value(\"${fieldName}\", change)",
                    entry("fieldName", name),
                    entry("methodPrefix", type.getSimpleName())
                );
            }
        }

        if (type == Boolean.class) {
            return "Boolean.valueOf(change.getBool())";
        }

        if (type == Float.class) {
            return "Float.valueOf(change.getDouble().floatValue())";
        }

        if (type == Double.class) {
            return "Double.valueOf(change.getDouble())";
        }

        if (isBoxedInteger(type)) {
            return format("ApplierOps#boxed${intType}Value(\"${fieldName}\", change)",
                entry("fieldName", name),
                entry("intType", type.getSimpleName())
            );
        }

        if (type == String.class) {
            return "change.getString()";
        }

        if (type == Optional.class) {
            val elementType = info.getGenericElements().get(0);
            return "change.getOptional(" + elementType.getCanonicalName() + ".class)";
        }

        if (type == OptionalInt.class) {
            return "change.getOptionalInt()";
        }

        if (type == OptionalLong.class) {
            return "change.getOptionalLong()";
        }

        if (type == OptionalDouble.class) {
            return "change.getOptionalDouble()";
        }

        if (type == Set.class) {
            val elementType = info.getGenericElements().get(0);
            return format("ApplierOps#applySetChange(\"${fieldName}\", ${fieldName}, change, ${elementType}.class)",
                entry("fieldName", info.getName()),
                entry("elementType", elementType.getCanonicalName())
            );
        }

        return format("(${fieldType}) change.get(${fieldType}.class)",
            entry("fieldType", info.getType().getCanonicalName())
        );
    }

    private static String generateFieldCheck(FieldInfo info) {
        return format("if (\"${fieldName}\".equals(change.name())) { ${fieldName} = ${applyCode}; break; }",
            entry("fieldName", info.getName()),
            entry("applyCode", generateApply(info))
        );
    }

    private static String generateCheck(Map.Entry<Integer, List<FieldInfo>> fieldsGroup) {
        val checks = fieldsGroup.getValue()
            .stream()
            .map(PatchApplierGenerator::generateFieldCheck)
            .collect(joining());

        return format("case ${hashCode}: {"
                    +     checks
                    + "   throw new UnexpectedFieldException(change.name());"
                    + "}",
            entry("hashCode", fieldsGroup.getKey())
        );
    }

    private static String generateTempFieldVariable(FieldInfo info) {
        return format("${fieldType} ${fieldName} = object.${getter}();",
            entry("fieldType", info.getType().getCanonicalName()),
            entry("fieldName", info.getName()),
            entry("getter", info.getGetterName())
        );
    }

    @SneakyThrows
    private static CtMethod generateApply(ClassInfo info, CtClass declaring) {
        val tempVariables = info.fieldsStream()
            .map(PatchApplierGenerator::generateTempFieldVariable)
            .joining();

        val fieldInfoByNameHashCode = info.fieldsStream()
            .groupingBy(fieldInfo -> fieldInfo.getName().hashCode());
        val fieldChecks = EntryStream.of(fieldInfoByNameHashCode)
            .map(PatchApplierGenerator::generateCheck)
            .joining();

        val constructorArgs = info.fieldsStream()
            .map(FieldInfo::getName)
            .joining(",");

        val source = format("public ${classType} apply(${classType} object, PatchProvider provider) {"
                          +      tempVariables
                          + "    java.util.List changes = provider.changes();"
                          + "    for (java.util.Iterator iter = changes.iterator(); iter.hasNext();) {"
                          + "        FieldChange change = (FieldChange) iter.next();"
                          + "        switch (change.name().hashCode()) {"
                          +              fieldChecks
                          + "            default: throw new UnexpectedFieldException(change.name());"
                          + "        }"
                          + "    }"
                          + "    return new ${classType}(" + constructorArgs + ");"
                          + "}",
            entry("classType", info.getType().getCanonicalName())
        );
        return CtNewMethod.make(source, declaring);
    }

    @SneakyThrows
    private static CtMethod generateApplyBridge(ClassInfo info, CtClass declaring) {
        val source = format("public Object apply(Object object, PatchProvider provider) {"
                          + "    return apply((${classType}) object, provider);"
                          + "}",
            entry("classType", info.getType().getCanonicalName())
        );
        return CtNewMethod.make(source, declaring);
    }

    @SneakyThrows
    CtClass generate(ClassInfo info) {
        val applierClass = pool.makeClass(info.getPatchApplierQualifiedName());

        val applierInterfaceType = PatchApplier.class.getCanonicalName();
        val signature = Utils.createGenericSignature(applierInterfaceType, info.getType().getCanonicalName());
        applierClass.addInterface(pool.get(applierInterfaceType));
        applierClass.setGenericSignature(signature.encode());

        applierClass.addMethod(generateApply(info, applierClass));
        applierClass.addMethod(generateApplyBridge(info, applierClass));

        return applierClass;
    }
}
