package ru.yandex.mail.diffusion.json;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeType;
import lombok.SneakyThrows;
import lombok.val;
import one.util.streamex.StreamEx;
import ru.yandex.mail.diffusion.patch.FieldChange;
import ru.yandex.mail.diffusion.patch.exception.UnexpectedTypeException;

import java.util.*;
import java.util.function.Function;

import static com.fasterxml.jackson.databind.node.JsonNodeType.*;
import static ru.yandex.mail.diffusion.json.JsonPatchCollector.*;

class JsonFieldChange implements FieldChange {
    private final JsonNode node;
    private final String name;
    private final ObjectMapper objectMapper;

    JsonFieldChange(JsonNode node, String name, ObjectMapper objectMapper) {
        val newNode = node.findValue(NEW);
        this.node = newNode == null ? node : newNode;
        this.name = name;
        this.objectMapper = objectMapper;
    }

    private static void checkNodeType(JsonNode node, String name, JsonNodeType... expectedTypes) {
        val nodeType = node.getNodeType();
        if (!StreamEx.of(expectedTypes).has(nodeType)) {
            throw new UnexpectedTypeException(name, expectedTypes, nodeType);
        }
    }

    private void checkSelfNodeType(JsonNodeType... expectedTypes) {
        checkNodeType(node, name, expectedTypes);
    }

    @SneakyThrows
    private <T> T readValue(JsonNode node, Class<T> type) {
        return objectMapper.treeToValue(node, type);
    }

    @Override
    public String name() {
        return name;
    }

    @Override
    public boolean getBool() {
        checkSelfNodeType(BOOLEAN);
        return node.asBoolean();
    }

    @Override
    public long getLong() {
        checkSelfNodeType(NUMBER);
        return node.asLong();
    }

    @Override
    public double getDouble() {
        checkSelfNodeType(NUMBER);
        return node.asDouble();
    }

    @Override
    public String getString() {
        checkSelfNodeType(STRING);
        return node.asText();
    }

    @Override
    @SneakyThrows
    public <T> T get(Class<T> type) {
        checkSelfNodeType(type.isEnum() ? STRING : OBJECT);
        return objectMapper.treeToValue(node, type);
    }

    @Override
    @SneakyThrows
    public <T> Optional<T> getOptional(Class<T> type) {
        checkSelfNodeType(STRING, NUMBER, BOOLEAN, NULL);
        return node.isNull() ? Optional.empty() : Optional.of(readValue(node, type));
    }

    @Override
    public OptionalInt getOptionalInt() {
        checkSelfNodeType(NUMBER, NULL);
        return node.isNull() ? OptionalInt.empty() : OptionalInt.of(node.asInt());
    }

    @Override
    public OptionalLong getOptionalLong() {
        checkSelfNodeType(NUMBER, NULL);
        return node.isNull() ? OptionalLong.empty() : OptionalLong.of(node.asLong());
    }

    @Override
    public OptionalDouble getOptionalDouble() {
        checkSelfNodeType(NUMBER, NULL);
        return node.isNull() ? OptionalDouble.empty() : OptionalDouble.of(node.asDouble());
    }

    private <T> Set<T> readSet(String setNodeName, Function<JsonNode, T> readValue) {
        checkSelfNodeType(OBJECT);

        val setNode = node.findValue(setNodeName);
        checkNodeType(setNode, name + '.' + setNodeName, ARRAY);

        val arrayNode = (ArrayNode) setNode;
        return StreamEx.of(arrayNode.elements())
            .map(readValue)
            .toSet();
    }

    @Override
    @SneakyThrows
    public <T> Set<T> getAdded(Class<T> type) {
        return readSet(ADD, text -> readValue(text, type));
    }

    @Override
    public <T> Set<T> getRemoved(Class<T> type) {
        return readSet(DEL, text -> readValue(text, type));
    }
}
