package ru.yandex.mail.diffusion.json;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.SneakyThrows;
import lombok.val;
import ru.yandex.mail.diffusion.FieldMatchListener;
import ru.yandex.mail.diffusion.FieldMatcher;
import ru.yandex.mail.diffusion.patch.DiffUtils;
import ru.yandex.mail.diffusion.patch.PatchCollector;

import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Stream;

public class JsonPatchCollector implements FieldMatchListener, AutoCloseable, PatchCollector<String> {
    static final String NEW = "new";
    static final String OLD = "old";
    static final String ADD = "add";
    static final String DEL = "del";

    private final ByteArrayOutputStream outputStream;
    private final ObjectMapper objectMapper;
    private final JsonGenerator generator;

    @SneakyThrows
    public JsonPatchCollector(ObjectMapper objectMapper) {
        outputStream = new ByteArrayOutputStream();
        this.objectMapper = objectMapper;
        generator = objectMapper.getFactory().createGenerator(outputStream, JsonEncoding.UTF8);
    }

    @SneakyThrows
    private <T> void writeValue(T value) {
        objectMapper.writeValue(generator, value);
    }

    @SneakyThrows
    private <T> void writeCollection(String fieldName, Stream<T> collection) {
        generator.writeArrayFieldStart(fieldName);
        collection.forEach(this::writeValue);
        generator.writeEndArray();
    }

    private interface Writer {
        void write() throws Exception;
    }

    @SneakyThrows
    private void writeChange(String fieldName, Writer writer) {
        generator.writeObjectFieldStart(fieldName);
        writer.write();
        generator.writeEndObject();
    }

    @Override
    @SneakyThrows
    public void close() {
        generator.close();
        outputStream.close();
    }

    @Override
    @SneakyThrows
    public <T> String collect(T oldObject, T newObject, FieldMatcher<T> matcher) {
        generator.writeStartObject();
        matcher.match(oldObject, newObject, this);
        generator.writeEndObject();

        close();
        return outputStream.toString(StandardCharsets.UTF_8);
    }

    @Override
    @SneakyThrows
    public void onMismatch(String fieldName, boolean oldValue, boolean newValue) {
        writeChange(fieldName, () -> {
            generator.writeBooleanField(OLD, oldValue);
            generator.writeBooleanField(NEW, newValue);
        });
    }

    @Override
    @SneakyThrows
    public void onMismatch(String fieldName, long oldValue, long newValue) {
        writeChange(fieldName, () -> {
            generator.writeNumberField(OLD, oldValue);
            generator.writeNumberField(NEW, newValue);
        });
    }

    @Override
    @SneakyThrows
    public void onMismatch(String fieldName, double oldValue, double newValue) {
        writeChange(fieldName, () -> {
            generator.writeNumberField(OLD, oldValue);
            generator.writeNumberField(NEW, newValue);
        });
    }

    @Override
    @SneakyThrows
    public void onMismatch(String fieldName, String oldValue, String newValue) {
        writeChange(fieldName, () -> {
            generator.writeStringField(OLD, oldValue);
            generator.writeStringField(NEW, newValue);
        });
    }

    @Override
    @SneakyThrows
    public <T> void onMismatch(String fieldName, T oldValue, T newValue) {
        writeChange(fieldName, () -> {
            generator.writeFieldName(OLD);
            objectMapper.writeValue(generator, oldValue);

            generator.writeFieldName(NEW);
            objectMapper.writeValue(generator, newValue);
        });
    }

    @Override
    @SneakyThrows
    public <T> void onMismatch(String fieldName, Set<T> oldValue, Set<T> newValue) {
        val diff = DiffUtils.findDiff(oldValue, newValue);

        generator.writeObjectFieldStart(fieldName);
        writeCollection(ADD, diff.getAdded().stream());
        writeCollection(DEL, diff.getRemoved().stream());
        generator.writeEndObject();
    }

    @SneakyThrows
    private <T> void writeValue(String fieldName, Optional<T> value) {
        if (value.isPresent()) {
            generator.writeFieldName(fieldName);
            objectMapper.writeValue(generator, value);
        } else {
            generator.writeNullField(fieldName);
        }
    }

    @Override
    @SneakyThrows
    public <T> void onMismatch(String fieldName, Optional<T> oldValue, Optional<T> newValue) {
        writeChange(fieldName, () -> {
            writeValue(OLD, oldValue);
            writeValue(NEW, newValue);
        });
    }

    @SneakyThrows
    private void writeValue(String fieldName, OptionalInt value) {
        if (value.isPresent()) {
            generator.writeNumberField(fieldName, value.getAsInt());
        } else {
            generator.writeNullField(fieldName);
        }
    }

    @Override
    @SneakyThrows
    public void onMismatch(String fieldName, OptionalInt oldValue, OptionalInt newValue) {
        writeChange(fieldName, () -> {
            writeValue(OLD, oldValue);
            writeValue(NEW, newValue);
        });
    }

    @SneakyThrows
    private void writeValue(String fieldName, OptionalLong value) {
        if (value.isPresent()) {
            generator.writeNumberField(fieldName, value.getAsLong());
        } else {
            generator.writeNullField(fieldName);
        }
    }

    @Override
    @SneakyThrows
    public void onMismatch(String fieldName, OptionalLong oldValue, OptionalLong newValue) {
        writeChange(fieldName, () -> {
            writeValue(OLD, oldValue);
            writeValue(NEW, newValue);
        });
    }

    @SneakyThrows
    private void writeValue(String fieldName, OptionalDouble value) {
        if (value.isPresent()) {
            generator.writeNumberField(fieldName, value.getAsDouble());
        } else {
            generator.writeNullField(fieldName);
        }
    }

    @Override
    @SneakyThrows
    public void onMismatch(String fieldName, OptionalDouble oldValue, OptionalDouble newValue) {
        writeChange(fieldName, () -> {
            writeValue(OLD, oldValue);
            writeValue(NEW, newValue);
        });
    }
}
