package ru.yandex.oauth;

import com.amazonaws.util.json.JSONException;
import com.amazonaws.util.json.JSONObject;
import com.spotify.sshagentproxy.AgentProxies;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.file.Paths;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;
import java.security.spec.InvalidKeySpecException;
import java.util.*;
import java.util.stream.Collectors;

import static com.amazonaws.auth.PEM.readPrivateKey;

@Slf4j
public class OAuth {
    private static String oauthHost = "https://oauth.yandex-team.ru/token";

    public OAuth() {
    }

    private List<String> genSignaturesFromAgentKeys(String data) throws IOException {
        try {
            val signatures = new ArrayList<String>();
            val agentProxy = AgentProxies.newInstance();
            val identities = agentProxy.list();
            identities.stream().filter(identity -> identity.getPublicKey().getAlgorithm().equals("RSA")).forEach(
                identity -> {
                    try {
                        val signedData = agentProxy.sign(identity, data.getBytes());
                        signatures.add(Base64.getUrlEncoder().encodeToString(signedData));
                    } catch (IOException e) {
                        log.error("Error signing data", e);
                    }
                });
            return signatures;
        } catch (RuntimeException e) {
            log.error("Error accessing ssh agent", e);
            return Collections.emptyList();
        }
    }

    private List<String> genSignaturesFromFiles(String data) throws IOException, NullPointerException {
        val signatures = new ArrayList<String>();
        val sshDir = new File(Paths.get(System.getProperty("user.home"), ".ssh").toString());
        for (val keyFile : Objects.requireNonNull(sshDir.listFiles())) {
            try {
                val pk = readPrivateKey(new FileInputStream(keyFile.toString()));

                val sig = Signature.getInstance("SHA1withRSA");
                sig.initSign(pk);
                sig.update(data.getBytes());

                val signatureBytes = sig.sign();
                signatures.add(Base64.getUrlEncoder().encodeToString(signatureBytes));
            } catch (NoSuchAlgorithmException | SignatureException | InvalidKeyException |
                    InvalidKeySpecException | IllegalArgumentException e) {
                log.error(e.toString());
            }
        }
        return signatures;
    }

    public List<String> getOauthToken(String oauthClientId, String oauthClientPassword) throws IOException {
        val login = System.getProperty("user.name");
        val ts = System.currentTimeMillis() / 1000;
        val data = String.format("%d%s%s", ts, oauthClientId, login);

        List<String> signatures = new ArrayList<>();

        signatures.addAll(genSignaturesFromAgentKeys(data));
        signatures.addAll(genSignaturesFromFiles(data));
        signatures = signatures.stream().distinct().collect(Collectors.toList());

        if (signatures.isEmpty()) {
            throw new RuntimeException("Can't find any rsa key");
        }

        val tokens = new ArrayList<String>();

        for (String signature : signatures) {
            val httpClient = HttpClients.createDefault();

            val post = new HttpPost(oauthHost);
            val postParameters = new ArrayList<BasicNameValuePair>();
            postParameters.add(new BasicNameValuePair("grant_type", "ssh_key"));
            postParameters.add(new BasicNameValuePair("client_id", oauthClientId));
            postParameters.add(new BasicNameValuePair("client_secret", oauthClientPassword));
            postParameters.add(new BasicNameValuePair("login", login));
            postParameters.add(new BasicNameValuePair("ts", Long.toString(ts)));
            postParameters.add(new BasicNameValuePair("ssh_sign", signature));

            try {
                post.setEntity(new UrlEncodedFormEntity(postParameters));
                val res = httpClient.execute(post);
                val obj = new JSONObject(EntityUtils.toString(res.getEntity()));
                val token = obj.getString("access_token");
                tokens.add(token);
            } catch (UnsupportedEncodingException | JSONException | ClientProtocolException e) {
                log.error(e.toString());
            }
        }
        return tokens.stream().distinct().collect(Collectors.toList());
    }
}
