package ru.yandex.mail.pglocal;

import lombok.Getter;
import lombok.val;
import org.postgresql.ds.PGSimpleDataSource;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;

public class Database {
    private static final int QUERY_TIMEOUT_SEC = 5;

    @Getter private final String name;
    @Getter private final Manager.DbOptions options;
    @Getter private final String connString;

    private Connection connect() throws SQLException {
        return DriverManager.getConnection(connString);
    }

    @FunctionalInterface
    public interface Fetcher<T> {
        T fetch(ResultSet resultSet) throws SQLException;
    }

    public Database(String name, Manager.DbOptions options) {
        this.name = name;
        this.options = options;
        this.connString = String.format("jdbc:postgresql://localhost:%d/%s?user=%s",
            options.getPort(), name, options.getUser());
    }

    public DataSource getSimpleDataSource() {
        val dataSource = new PGSimpleDataSource();
        dataSource.setServerName("localhost");
        dataSource.setPortNumber(options.getPort());
        dataSource.setUser(options.getUser());
        dataSource.setSsl(false);
        dataSource.setDatabaseName(name);
        return dataSource;
    }

    public void execute(String query) {
        execute(query, QUERY_TIMEOUT_SEC);
    }

    public void execute(String query, int timeoutSec) {
        try (val connection = connect();
             val statement = connection.createStatement()) {
            statement.setQueryTimeout(timeoutSec);
            statement.execute(query);
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    public <T> T fetch(String query, Fetcher<T> fetcher) {
        return fetch(query, QUERY_TIMEOUT_SEC, fetcher);
    }

    public <T> T fetch(String query, int timeoutSec, Fetcher<T> fetcher) {
        try (val connection = connect();
             val statement = connection.createStatement()) {
            statement.setQueryTimeout(timeoutSec);
            val resultSet = statement.executeQuery(query);
            return fetcher.fetch(resultSet);
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }
}
