package ru.yandex.mail.pglocal;

import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.SystemUtils;
import org.apache.commons.text.StringSubstitutor;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.ServerSocket;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static java.nio.file.StandardOpenOption.*;
import static ru.yandex.mail.pglocal.PgExecutor.PG_CONFIG;

@Slf4j
public class Manager {
    private static final String REPLICA_USER = "replica";

    @Value
    public static class DbOptions {
        int port;
        String user;

        public static DbOptions withRandomPort(String user) {
            try (val socket = new ServerSocket(0)) {
                return new DbOptions(socket.getLocalPort(), user);
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    @Value
    public static class SlaveOptions {
        DbOptions db;
        String applicationName;
        String database;
    }

    @Value
    public static class MasterOptions {
        DbOptions db;
        SynchronousCommit synchronousCommit;
        List<String> synchronousStandbyNames;
    }

    private final Path pgPath;
    private final PgExecutor executor;

    public Manager(BinarySource binarySource) {
        pgPath = binarySource.fetch();
        executor = new PgExecutor(pgPath, pgPath, log);
    }

    public Server startNewMaster(Path data, MasterOptions options) {
        cleanupData(data.toFile());
        executor.initDb(data, options.db.user);
        preparePgConfig(data, options);
        preparePgHbaConfig(data);

        val server = new Server(pgPath, data, options.db, ServerType.MASTER);
        server.start();
        server.attachDefaultDatabase().execute("CREATE USER " + REPLICA_USER + " REPLICATION LOGIN");
        return server;
    }

    public Server startMaster(Path data, MasterOptions options) {
        preparePgConfig(data, options);
        preparePgHbaConfig(data);
        val server = new Server(pgPath, data, options.db, ServerType.MASTER);
        server.start();
        return server;
    }

    public Server startNewSlave(Path data, int masterPort, SlaveOptions options) {
        cleanupData(data.toFile());
        executor.makeBackup(masterPort, data);
        return startSlave(data, masterPort, options);
    }

    public Server startSlave(Path data, int masterPort, SlaveOptions options) {
        preparePgConfig(data, new MasterOptions(options.db, SynchronousCommit.OFF, Collections.emptyList()));
        prepareReplicaConfig(data, masterPort, options);
        val server = new Server(pgPath, data, options.db, ServerType.SLAVE);
        server.start();
        return server;
    }

    private void preparePgConfig(Path data, MasterOptions options) {
        val config = new HashMap<String, String>() {{
            put("port", String.valueOf(options.db.port));
            put("synchronous_commit", options.synchronousCommit.configValue());

            val names = options.synchronousStandbyNames
                .stream()
                .map(name -> '\'' + name + '\'')
                .collect(Collectors.joining(","));
            put("synchronous_standby_names", names.isEmpty() ? "'*'" : names);
            put("unix_socket_directories", SystemUtils.IS_OS_UNIX ? "unix_socket_directories = '/tmp'" : "");
        }};

        copyTemplateFile("pg.conf", data.resolve(PG_CONFIG), config);
    }

    private void preparePgHbaConfig(Path data) {
        val hbaTemplate = SystemUtils.IS_OS_WINDOWS ? "pg_hba_win.conf" : "pg_hba_unix.conf";
        copyTemplateFile(hbaTemplate, data.resolve("pg_hba.conf"), Collections.emptyMap());
    }

    private void prepareReplicaConfig(Path data, int masterPort, SlaveOptions options) {
        val config = new HashMap<String, String>(){{
            put("port", String.valueOf(masterPort));
            put("dbname", options.database);
            put("app_name", options.applicationName);
        }};
        copyTemplateFile("recovery.conf", data.resolve("recovery.conf"), config);
    }

    private void copyTemplateFile(String name, Path destination, Map<String, String> values) {
        try {
            val resourceStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(name);
            if (resourceStream == null) {
                log.error("Resource '{}' not found", name);
                throw new RuntimeException("Resource '" + name + "' not found");
            }

            val substitutor = new StringSubstitutor(values, "{", "}");

            val template = IOUtils.toString(resourceStream);
            val text = substitutor.replace(template);
            Files.write(destination, text.getBytes(), WRITE, TRUNCATE_EXISTING, CREATE);
        } catch (IOException e) {
            log.error("Error reading template file '{}'", name);
            throw new UncheckedIOException(e);
        }
    }

    private void cleanupData(File dataDir) {
        if (dataDir.exists()) {
            try {
                FileUtils.forceDelete(dataDir);
            } catch (IOException e) {
                log.error("Data folder '" + dataDir + "' cleanup failed", e);
                throw new UncheckedIOException(e);
            }
        }
    }
}
