package ru.yandex.mail.pglocal;

import lombok.val;
import org.apache.commons.io.FileUtils;
import org.flywaydb.core.Flyway;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.mail.pglocal.Manager.DbOptions;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Path;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;

public class Server {
    private final Path data;
    private final DbOptions options;
    private final ServerType type;
    private final Logger log;
    private final PgExecutor executor;
    private AtomicBoolean running;

    public Server(Path pgPath, Path data, DbOptions options, ServerType type) {
        this.data = data;
        this.options = options;
        this.type = type;
        this.running = new AtomicBoolean(true);
        log = LoggerFactory.getLogger("pg_instance_" + data.getFileName());
        executor = new PgExecutor(pgPath, data, log);
        registerServerStop();
    }

    public DbOptions getOptions() {
        return options;
    }

    public DatabaseStatus status() {
        return executor.status(data);
    }

    public void start() {
        if (status() == DatabaseStatus.STOPPED) {
            executor.start(data, options.getPort(), type);
        }
    }

    public void stop() {
        running.set(false);
        if (status() == DatabaseStatus.STARTED) {
            executor.stop(data, type);
        }
    }

    public Database createDatabase(String name, String user, Optional<MigrationSource> migrations) {
        executor.createDb(name, options.getPort(), user);
        migrations.ifPresent(source -> applyMigrations(name, options, source));
        return new Database(name, options);
    }

    public Database attachDatabase(String name) {
        return new Database(name, options);
    }

    public Database attachDefaultDatabase() {
        return attachDatabase("postgres");
    }

    public void purge() {
        try {
            stop();
            FileUtils.forceDelete(data.toFile());
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void applyMigrations(String database, DbOptions options, MigrationSource migrations) {
        log.info("Apply migrations");
        val flyway = new Flyway();
        val connString = String.format("jdbc:postgresql://localhost:%d/%s", options.getPort(), database);
        flyway.setDataSource(connString, options.getUser(), "");
        flyway.setLocations(migrations.flywayPaths());
        val applied = flyway.migrate();
        log.info("{} migrations applied", applied);
    }

    private void registerServerStop() {
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            try {
                if (running.get()) {
                    stop();
                }
            } catch (Exception ignored) {
            }
        }));
    }
}
