package ru.yandex.autotests.innerpochta.tomita.core;

import ch.ethz.ssh2.Connection;
import ch.ethz.ssh2.Session;
import ch.ethz.ssh2.StreamGobbler;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.fluent.Executor;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import ru.yandex.autotests.innerpochta.tomita.core.beans.Component;
import ru.yandex.autotests.innerpochta.tomita.core.beans.DataBean;
import ru.yandex.autotests.innerpochta.wmi.core.oper.Message;

import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.internet.MimeMessage;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Properties;

import static ch.lambdaj.collection.LambdaCollections.with;
import static com.google.common.base.Charsets.UTF_8;
import static com.google.common.base.Joiner.on;
import static javax.mail.Session.getDefaultInstance;
import static javax.ws.rs.core.HttpHeaders.CONTENT_TYPE;
import static javax.ws.rs.core.UriBuilder.fromUri;
import static org.apache.commons.lang.StringUtils.defaultIfEmpty;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.apache.http.client.fluent.Request.Post;
import static org.apache.http.entity.ContentType.getOrDefault;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.innerpochta.tomita.core.TomitaProps.tomitaProps;
import static ru.yandex.autotests.innerpochta.tomita.core.converters.ToObjectConverter.wrapAndAddTo;
import static ru.yandex.autotests.innerpochta.tomita.core.opers.Factextract.selectBy;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.api;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;
import static ru.yandex.autotests.innerpochta.wmi.core.filter.log.LoggerFilterBuilder.log;
import static ru.yandex.autotests.innerpochta.wmi.core.obj.MessageObj.getMsgWithContentFlag;
import static ru.yandex.autotests.innerpochta.wmi.core.obj.Obj.XMLVERSION_DARIA2;
import static ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule.auth;
import static ru.yandex.autotests.innerpochta.wmi.core.utils.CommonUtils.getMidsOfAllMsgsInFolder;
import static ru.yandex.qatools.elliptics.ElClient.elliptics;

/**
 * User: lanwen
 * Date: 07.04.14
 * Time: 21:25
 */
public final class ContentGenUtils {

    public static final String NO_SUBJECT = "(Без темы)";

    public static String getAnyTextFromMessage(Part part) throws MessagingException, IOException {
        if (part.isMimeType("text/html") || part.isMimeType("text/plain")) {
            try {
                return (String) part.getContent();
            } catch (UnsupportedEncodingException e) {
                //Use default enc on parse exception
                part.setHeader(CONTENT_TYPE, "text/plain; charset=UTF-8");
                return (String) part.getContent();
            } catch (IOException e) {
                part.setHeader(CONTENT_TYPE, "text/plain; charset=UTF-8");
                part.removeHeader("Content-Transfer-Encoding");
                //Use default enc on parse exception
                return (String) part.getContent();
            }
        }
        StringBuilder parseResult = new StringBuilder();

        if (part.isMimeType("multipart/alternative")) {
            // prefer html text over plain text
            Multipart mp = (Multipart) part.getContent();
            String text = null;
            for (int i = 0; i < mp.getCount(); i++) {
                Part bp = mp.getBodyPart(i);
                if (bp.isMimeType("text/plain")) {
                    if (text == null) {
                        text = getAnyTextFromMessage(bp);
                    }
                } else if (bp.isMimeType("text/html")) {
                    String s = getAnyTextFromMessage(bp);
                    if (isNotBlank(s)) {
                        text = s;
                    }
                } else {
                    parseResult.append(getAnyTextFromMessage(bp));
                }
            }
            parseResult.append(text);
        } else if (part.isMimeType("multipart/*")) {
            Multipart mp = (Multipart) part.getContent();
            for (int i = 0; i < mp.getCount(); i++) {
                parseResult.append(getAnyTextFromMessage(mp.getBodyPart(i)));
            }
        } else if (part.isMimeType("message/delivery-status")) {
            String str = IOUtils.toString(part.getInputStream());
            parseResult.append(str);
        }

        return parseResult.toString();
    }

    public static MimeMessage mimeMessageFor(DataBean bean, DefaultHttpClient authHc) throws IOException {
        return Executor.newInstance(authHc).execute(Post(fromUri(props().betaURI())
                .path("/neo2/handlers/message-source/{mid}/yandex_email.eml").build(bean.mid())))
                .handleResponse(new ResponseHandler<MimeMessage>() {
                    @Override
                    public MimeMessage handleResponse(HttpResponse httpResponse) throws IOException {
                        try {
                            Charset charset = defaultIfNull(getOrDefault(httpResponse.getEntity()).getCharset(), UTF_8);
                            LogManager.getLogger(ContentGenUtils.class).info(charset);
                        } catch (IllegalCharsetNameException e) {
                            LogManager.getLogger(ContentGenUtils.class).error("Can't parse: " + e.getMessage(), e);
                        }

                        try(InputStream content = httpResponse.getEntity().getContent()) {
                            return new MimeMessage(getDefaultInstance(new Properties()), content);
                        } catch (MessagingException e) {
                            LogManager.getLogger(ContentGenUtils.class).error(e);
                            return new MimeMessage(getDefaultInstance(new Properties()));
                        }
                    }
                });
    }


    public static String executeCommAndResturnResultAsString(Connection conn, String comm)
            throws IOException {

        Logger log = Logger.getLogger(ContentGenUtils.class);

        Session sess = conn.openSession();
        log.debug(comm);
        sess.execCommand(comm);
        InputStream stdout = new StreamGobbler(sess.getStdout());
        InputStream stderr = new StreamGobbler(sess.getStderr());
        String out = IOUtils.toString(stdout);
        String err = IOUtils.toString(stderr);
        if (isNotEmpty(err)) {
            log.error(err);
        }
        sess.close();
        return out;
    }


    public static List<Object[]> messagesFor(Component component) throws IOException {
        DefaultHttpClient authClnt = auth().with(tomitaProps(component).getLoginGroup()).login().authHC();

        final List<String> goodMids = getMidsOfAllMsgsInFolder(tomitaProps(component).getFidGood(), authClnt);
//        final List<String> goodMids = newArrayList("2360000002332626984");
//        final List<String> badMids = getMidsOfAllMsgsInFolder(tomitaProps(component).getFidBad(), authClnt);

        List<Object[]> list = new ArrayList<>();
        list.addAll(with(goodMids).convert(wrapAndAddTo(authClnt, component, tomitaProps(component).getFidGood())));
//        list.addAll(with(badMids).convert(wrapAndAddTo(authClnt, component, tomitaProps(component).getFidBad())));

        return list;
    }

    public static String respUrl(DataBean beanToSend) throws IOException {
        String prod = selectBy(beanToSend).setHost(tomitaProps(beanToSend.component()).getTomitaProduction())
                .log(log().nothing())
                .post().via(new DefaultHttpClient()).toString();
        return elliptics().path(tomitaProps(beanToSend.component()).datapath()).randomize()
                .name(".prod.json").put(prod).get().url();
    }


    public static DataBean populate(DataBean bean, DefaultHttpClient authHc) throws IOException, MessagingException {
        Logger log = LogManager.getLogger(ContentGenUtils.class);
        MimeMessage msg = mimeMessageFor(bean, authHc);

        Message resp = api(Message.class).params(getMsgWithContentFlag(bean.mid())
                .setXmlVersion(XMLVERSION_DARIA2))
                .log(log().nothing())
                .post().via(authHc);

        log.info("Достаем текстовую информацию из письма с темой: " + msg.getSubject());
        String content = on(". ").join(defaultIfEmpty(msg.getSubject(), NO_SUBJECT), getAnyTextFromMessage(msg));
        String path = elliptics().path(tomitaProps(bean.component()).datapath()).randomize()
                .name(".data.txt").put(content)
                .check().url(notNullValue(String.class)).log(log)
                .pattern("Извлеченный текст из данного письма хранится по адресу: {url}")
                .get().fullpath();

        return DataBean.fromBean(bean).sender(resp.fromEmail())
                .path(path)
                .originalMessageId(resp.getMessageIdTag())
                .types(resp.types())
                .date(resp.infoTimestampSec())
                .firstline(resp.getFirstlineText());
    }
}
