package ru.yandex.autotests.innerpochta.tomita.core.rules;

import com.google.gson.Gson;
import org.junit.rules.TestWatcher;
import org.junit.runner.Description;
import ru.yandex.autotests.innerpochta.tomita.core.beans.Component;
import ru.yandex.autotests.innerpochta.tomita.core.beans.DataBean;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static ch.lambdaj.Lambda.having;
import static ch.lambdaj.Lambda.on;
import static ch.lambdaj.Lambda.select;
import static ch.lambdaj.collection.LambdaCollections.with;
import static java.util.Arrays.asList;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.innerpochta.tomita.core.TomitaProps.tomitaProps;
import static ru.yandex.autotests.innerpochta.tomita.core.converters.DataBeanToUrlConverter.urlExtractor;
import static ru.yandex.qatools.elliptics.ElClient.elliptics;

/**
 * User: lanwen
 * Date: 07.04.14
 * Time: 21:43
 */
public class TomitaDataPublisher extends TestWatcher {
    private List<Component> components;

    private List<DataBean> beans = new ArrayList<>();
    private Map<DataBean, String> responses = new HashMap<>();
    private Map<DataBean, String> reqUrls = new HashMap<>();
    private boolean cantSave = true;

    private TomitaDataPublisher(List<Component> components) {
        this.components = components;
    }

    public static TomitaDataPublisher withComponents(Component... components) {
        return new TomitaDataPublisher(asList(components));
    }

    public static TomitaDataPublisher withComponents(List<Component> components) {
        return new TomitaDataPublisher(components);
    }

    @Override
    protected void succeeded(Description description) {
        if (cantSave) {
            return;
        }

        for (Component component : components) {

            if (tomitaProps(component).needExtract()) {
                List<DataBean> beansFiltered = select(beans,
                        having(on(DataBean.class).component(), equalTo(component)));
                save(component, beansFiltered);
                export(component, beansFiltered);
            }
        }
    }

    public void canSaveNow() {
        this.cantSave = false;
    }

    public void addBean(DataBean bean) {
        beans.add(bean);
    }

    public void putRespUrl(DataBean bean, String resp) {
        responses.put(bean, resp);
    }

    public void putReqUrl(DataBean bean, String req) {
        reqUrls.put(bean, req);
    }


    public List<Component> components() {
        return components;
    }


    private void save(Component component, List<DataBean> beans) {
        if (beans.size() < 1) {
            return;
        }
        // Для аквы
        elliptics().path(tomitaProps(component).datapath())
                .name(tomitaProps(component).dataname())
                .update(new Gson().toJson(beans));
    }

    private void export(Component component, List<DataBean> beans) {
        if (beans.size() < 1) {
            return;
        }

        // Экспорт для томиты
        elliptics().path(tomitaProps(component).datapath())
                .name(tomitaProps(component).dataexport()).delete();
        String join = with(beans).convert(urlExtractor(responses, reqUrls)).join("\n");
        elliptics().path(tomitaProps(component).datapath())
                .name(tomitaProps(component).dataexport())
                .put(join);
    }
}