package ru.yandex.autotests.innerpochta.tomita.prepare;

import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.junit.ClassRule;
import org.junit.FixMethodOrder;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.MethodSorters;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.innerpochta.tomita.core.beans.Component;
import ru.yandex.autotests.innerpochta.tomita.core.beans.DataBean;
import ru.yandex.autotests.innerpochta.tomita.core.rules.TomitaDataCollector;
import ru.yandex.autotests.innerpochta.tomita.core.rules.TomitaDataPublisher;
import ru.yandex.autotests.innerpochta.wmi.core.rules.local.SshLocalPortForwardingRule;
import ru.yandex.autotests.plugins.testpers.LogConfigRule;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.allure.annotations.Title;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.Collection;

import static com.google.common.base.Joiner.on;
import static com.jayway.restassured.RestAssured.given;
import static java.lang.String.format;
import static org.apache.commons.lang3.RandomStringUtils.randomNumeric;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.hamcrest.Matchers.*;
import static org.junit.Assume.assumeNoException;
import static org.junit.Assume.assumeThat;
import static ru.yandex.autotests.innerpochta.tomita.core.ContentGenUtils.getAnyTextFromMessage;
import static ru.yandex.autotests.innerpochta.tomita.core.ContentGenUtils.messagesFor;
import static ru.yandex.autotests.innerpochta.tomita.core.ContentGenUtils.mimeMessageFor;
import static ru.yandex.autotests.innerpochta.tomita.core.TomitaProps.tomitaProps;
import static ru.yandex.autotests.innerpochta.tomita.core.beans.Component.SANITIZER;
import static ru.yandex.autotests.innerpochta.tomita.core.beans.DataBean.fromMid;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;
import static ru.yandex.autotests.innerpochta.wmi.core.rules.mock.MockRule.localPortForMocking;
import static ru.yandex.autotests.innerpochta.wmi.core.rules.local.SshLocalPortForwardingRule.viaRemoteHost;
import static ru.yandex.qatools.elliptics.ElClient.elliptics;

/**
 * User: alex89
 * Date: 24.07.13
 * https://st.yandex-team.ru/AUTOTESTPERS-60
 * Список всех адресов смотреть здесь:
 * http://qa.yandex-team.ru/storage/get/tomita/testdata/pathlist-lamertester2210001110028383812.txt
 * http://qa.yandex-team.ru/storage/get/tomita/testdata/pathlist-eticket.type2170000570000032357.txt
 */

@Aqua.Test
@Title("Генератор тестовых данных - Санитайзер")
@Description("Подготовка бинов для санитайзера")
@Features({"TOMITA", "GENERATOR"})
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
@RunWith(Parameterized.class)
@Stories("Санитайзер")
public class TomitaDataPrepareSanitizer {
    private static Logger log = LogManager.getLogger(TomitaDataPrepareSanitizer.class);

    public static final String REQ_PATH_SPAM = "/spam";
    public static final String REQ_PATH_PR_HTTPS = "/pr_https";


    @Parameterized.Parameters(name = "{1}-{3}")
    public static Collection<Object[]> data() throws IOException {
        return messagesFor(SANITIZER).subList(0, 300);
    }


    private DefaultHttpClient authHc;
    private DataBean bean;

    private static String path;
    private boolean fileOnRemote = true;

    public TomitaDataPrepareSanitizer(DefaultHttpClient hc, Component component, String fid, String mid) {
        this.authHc = hc;
        bean = fromMid(mid).fid(fid).component(component);
    }


    // TODO: 2360000002332627125


    @ClassRule
    public static TomitaDataPublisher collect = TomitaDataPublisher.withComponents(SANITIZER);

    public TomitaDataCollector upload = TomitaDataCollector.withPublisher(collect);

    @Rule
    public RuleChain aqua = new LogConfigRule().around(upload);

    @ClassRule
    public static SshLocalPortForwardingRule portForwarding = viaRemoteHost(props().betaURI())
             .forwardTo(props().betaURI())
            .withForwardToPort(tomitaProps(SANITIZER).sanitizerUri().getPort()).onLocalPort(localPortForMocking());

    @Test
    public void aGetData() throws IOException, MessagingException {
        assumeThat("Экстрактинг санитайзера должен быть включен", tomitaProps(bean.component())
                .needExtract(), is(true));
        collect.canSaveNow();

        MimeMessage msg = mimeMessageFor(bean, authHc);

        log.info("Достаем текстовую информацию из письма с темой: " + msg.getSubject());
        String content;

        try {
            content = URLEncoder.encode(getAnyTextFromMessage(msg), "UTF-8");
        } catch (OutOfMemoryError error) {
            fileOnRemote = false;
            assumeNoException(error);
            return;
        }

        if(isEmpty(content)) {
            fileOnRemote = false;
            return;
        }

        String requestBody = on("&text=")
                .join(format("id=%s-%s-%s", randomNumeric(5), randomNumeric(5), "mdb150"), content);

        path = elliptics().path(tomitaProps(bean.component()).datapath()).randomize()
                .name(".data.htm").put(requestBody)
                .check().url(notNullValue(String.class)).log(log)
                .pattern("Извлеченный текст из данного письма хранится по адресу: {url}")
                .get().fullpath();


        log.trace("Send file...");
    }


    @Test
    public void extractSanitizerPRHttps() throws Exception {
        assumeThat("Экстрактинг санитайзера должен быть включен", tomitaProps(bean.component())
                .needExtract(), is(true));
        assumeThat("Должны были успешно залить файл", fileOnRemote, is(true));

        DataBean prhttps = fromMid(bean.mid()).fid(bean.fid()).component(bean.component()).firstline("pr_https");
        upload.setData(prhttps.path(path));

        assumeThat("Должны были успешно залить файл", trimToNull(prhttps.data()), notNullValue());
        /*
        PR_HTTPS
         */

        try(InputStream result = given()
                .body(prhttps.data()).baseUri(portForwarding.local().toString())
                .post(REQ_PATH_PR_HTTPS).asInputStream()) {

            String urlPrHttps = elliptics().path(tomitaProps(bean.component()).datapath())
                    .randomize().name(".pr_https.prod.txt")
                    .put(result)
                    .log().asLink().get().url();

            upload.reqUrl(REQ_PATH_PR_HTTPS).respDataUrl(urlPrHttps);
        }
    }


    @Test
    public void extractSanitizerSpam() throws Exception {
        assumeThat("Экстрактинг санитайзера должен быть включен", tomitaProps(bean.component())
                .needExtract(), is(true));
        assumeThat("Должны были успешно залить файл", fileOnRemote, is(true));

        DataBean spam = fromMid(bean.mid()).fid(bean.fid()).component(bean.component()).firstline("spam");
        upload.setData(spam.path(path));

        assumeThat("Должны были успешно залить файл", trimToNull(spam.data()), notNullValue());
        /*
        SPAM
         */

        try(InputStream result = given()
                .body(spam.data()).baseUri(portForwarding.local().toString())
                .post(REQ_PATH_SPAM).asInputStream()) {

            String urlSpam = elliptics().path(tomitaProps(bean.component()).datapath())
                    .randomize().name(".spam.prod.txt")
                    .put(result)
                    .log().asLink().get().url();

            upload.reqUrl(REQ_PATH_SPAM).respDataUrl(urlSpam);
        }

    }


}

