package ru.yandex.autotests.innerpochta.wmi.adapter;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.hamcrest.Matcher;
import org.json.JSONException;
import org.json.JSONObject;
import ru.yandex.autotests.innerpochta.beans.yplatform.Folder;
import ru.yandex.autotests.innerpochta.tests.unstable.TestMessage;
import ru.yandex.autotests.innerpochta.wmi.core.api.MbodyApi;
import ru.yandex.autotests.innerpochta.wmi.core.matchers.labels.MidsWithLabelMatcher;
import ru.yandex.autotests.innerpochta.wmi.core.obj.hound.FilterSearchObj;
import ru.yandex.autotests.innerpochta.wmi.core.obj.hound.FoldersObj;
import ru.yandex.autotests.innerpochta.wmi.core.obj.hound.MessagesByFolderObj;
import ru.yandex.autotests.innerpochta.wmi.core.oper.SettingsFolderClear;
import ru.yandex.autotests.innerpochta.wmi.core.oper.hound.FilterSearchCommand;
import ru.yandex.autotests.innerpochta.wmi.core.oper.hound.Folders;
import ru.yandex.autotests.innerpochta.wmi.core.oper.mops.Mops;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;
import ru.yandex.autotests.innerpochta.wmi.core.rules.mops.CleanMessagesMopsRule;
import ru.yandex.autotests.innerpochta.wmi.core.rules.mops.DeleteFoldersRule;
import ru.yandex.autotests.innerpochta.wmi.core.rules.mops.DeleteLabelsMopsRule;
import ru.yandex.autotests.innerpochta.wmi.core.utils.FolderList;
import ru.yandex.junitextensions.rules.loginrule.Credentials;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.concurrent.TimeUnit.SECONDS;
import static java.util.function.Function.identity;
import static javax.mail.Session.getDefaultInstance;
import static org.apache.commons.io.IOUtils.toInputStream;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.core.IsEqual.equalTo;
import static ru.yandex.autotests.innerpochta.matchers.MessageContentMatcher.hasContent;
import static ru.yandex.autotests.innerpochta.wmi.core.api.CommonApiSettings.shouldBe;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.api;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;
import static ru.yandex.autotests.innerpochta.wmi.core.filter.log.LoggerFilterBuilder.log;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.IsNot.not;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.WaitForMatcherDecorator.withWaitFor;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.messages.IsThereMessagesMatcher.hasMsgsIn;
import static ru.yandex.autotests.innerpochta.wmi.core.mops.MopsResponses.okSync;
import static ru.yandex.autotests.innerpochta.wmi.core.obj.SettingsFolderClearObj.purgeFid;
import static ru.yandex.autotests.innerpochta.wmi.core.obj.hound.LabelsObj.empty;
import static ru.yandex.autotests.innerpochta.wmi.core.oper.hound.MessagesByFolder.messagesByFolder;
import static ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule.auth;


/**
 * User: alex89
 * Date: 17.05.13
 */

@SuppressWarnings("ALL")
public class WmiAdapterUser {
    private Logger log = LogManager.getLogger(this.getClass());
    private static final String PROD_HOST = "https://mail.yandex.ru";
    private static final String ADAPTER_BASE_HOST = props().b2bUri().toString();
    private static final String YANDEX_TEAM_DOMAIN_PART = "yandex-team.";
    private static final int DEFAULT_TIMEOUT = 30;

    private String mailBoxLogin = "пустой логин";
    private String mailBoxPwd = "пустой пароль";
    private HttpClientManagerRule auth;
    private FolderList folderList;
    private String uid = "";
    private String suid = "";
    private String db = "pq";
    private String currentFid = "";//fid папки, где осуществляются действия проверок.

    public WmiAdapterUser(String mailBoxLogin, String mailBoxPwd) {
        this.mailBoxLogin = mailBoxLogin;
        this.mailBoxPwd = mailBoxPwd;
        setWMIProperty(mailBoxLogin);
        auth = auth().with(mailBoxLogin, mailBoxPwd).login();
        uid = auth.account().uid();
        suid = auth.account().suid();
        folderList = new FolderList(auth);
        currentFid = folderList.defaultFID();
    }

    public WmiAdapterUser(Credentials mailBoxAccount) {
        this(mailBoxAccount.getLogin(), mailBoxAccount.getPassword());
    }

    public static void setWMIProperty(String mailBoxLogin) {
        props().reset();
        if (mailBoxLogin.contains(YANDEX_TEAM_DOMAIN_PART)) {
            System.setProperty("system.testing.scope", "webcorp");
        } else {
            System.setProperty("system.testing.scope", "pg");
            System.setProperty("selenium.baseURL", ADAPTER_BASE_HOST);
        }
    }

    public String getLogin() {
        return mailBoxLogin;
    }

    public String getMdb() {
        return db;
    }

    public String getUid() {
        return uid;
    }

    public String getSuid() {
        return suid;
    }

    public HttpClientManagerRule getAuth() {
        return auth;
    }

    public String getFid(String folderPath) {
        try {
            Map<String, Folder> folders = new FolderList(getAuth()).folders();
            for (String fid : folders.keySet()) {
                if (folders.get(fid).getName().equals(folderPath)) {
                    return fid;
                }
            }
            return null;
        } catch (Exception var3) {
            throw new RuntimeException("Could not get fid by folder path: " + var3.getMessage());
        }
    }

    public String getLid(String labelName) {
        try {
            ru.yandex.autotests.innerpochta.wmi.core.oper.hound.Labels labels =
                    api(ru.yandex.autotests.innerpochta.wmi.core.oper.hound.Labels.class)
                            .setHost(props().houndUri())
                            .params(empty().setUid(uid))
                            .get()
                            .via(auth);
            return labels.lidByName(labelName);
        } catch (Exception e) {
            throw new RuntimeException("Could not get lid by label name: " + e.getMessage());
        }
    }

    //Получаем lid-ы у меток
    private List<String> getLabelIdsByNames(List<String> labelNames) {
        List<String> lidsList = new ArrayList<String>();
        for (String labelName : labelNames) {
            String labelId = getLid(labelName);
            log.info("lid=" + labelId);
            lidsList.add(labelId);
        }
        return lidsList;
    }

    public WmiAdapterUser inFolderWithFid(String fid) {
        currentFid = fid;
        return this;
    }

    /**
     * @param folderPath -- путь до папки, например
     *                   "Отправленные" - системная папка Отправленные
     *                   "Archive2|fold" - папка fold в пользовательской папке Archive2
     *                   "Отправленные|fold2" - папка fold2 в системной папке Отправленные
     *                   "Archive|fold3" - папка fold3 в системной папке Archive
     * @return WmiAdapterUser
     */

    public WmiAdapterUser inFolder(String folderPath) {
        currentFid = getFid(folderPath);
        return this;
    }

    public void isLogIn() throws RuntimeException, IOException {
        if (!auth.authHC().getCookieStore().getCookies().toString().contains("Session_id")) {
            throw new RuntimeException(format("Не удалось залогиниться к пользователю:{%s %s}",
                    mailBoxLogin, mailBoxPwd));
        }
    }

    /**********
     * Создание меток и папок
     *********/

    /**
     * С помощью WMI создаём метку с названием  labelName
     * <p/>
     *
     * @param labelName - название метки
     * @return id метки
     * @throws java.io.IOException *
     */
    public String createLabel(String labelName) throws IOException {
        return Mops.newLabelByName(auth, labelName);
    }


    /**
     * С помощью WMI создаём папку с названием  folderName
     * <p/>
     *
     * @param folderName - название папки
     * @return id папки
     * @throws java.io.IOException *
     */
    public String createFolder(String folderName) throws IOException {
        return Mops.newFolder(auth, folderName);
    }

    /***********
     * Удаление всего и вся
     **********/

    /**
     * Удаляет все письма в ящике
     *
     * @return WmiAdapterUser
     */
    public WmiAdapterUser clearAll() {
        CleanMessagesMopsRule.with(auth).allfolders().call();
        log.info("Почистили все папки от писем");
        return this;
    }

    /**
     * Удаляет все письма во входящих  почтового ящика
     *
     * @return WmiAdapterUser
     */
    public WmiAdapterUser clearDefaultFolder() {
        CleanMessagesMopsRule.with(auth).inbox().call();
        log.info("Почистили папку\"Входяшие\" в ящике " + mailBoxLogin);
        return this;
    }

    /**
     * Удаляет все письма во входящих  почтового ящика
     *
     * @return WmiAdapterUser
     */
    public WmiAdapterUser clearDefaultFolderWithProduction() {
        api(SettingsFolderClear.class).setHost(PROD_HOST)
                .log(log().onlyIfErrorOrXmlWithError())
                .params(purgeFid(folderList.defaultFID()))
                .post().via(auth.authHC());
        log.info("Почистили папку\"Входяшие\" через продакшн в ящике " + mailBoxLogin);
        return this;
    }

    /**
     * Удаляет все письма в указанной папке  почтового ящика
     *
     * @return WmiAdapterUser
     */
    public WmiAdapterUser clearFolder() {
        CleanMessagesMopsRule.with(auth).allfolders().call();
        log.info("Почистили все папки от писем");
        return this;
    }


    /**
     * С помощью WMI в почтовом ящике {mailBoxLogin, mailBoxPwd}
     * удаляет все пользовательские папки.
     *
     * @return WmiAdapterUser
     */
    public WmiAdapterUser deleteAllFolders() {
        DeleteFoldersRule.with(auth).all().call();
        log.info("Удалили все папки в ящике " + mailBoxLogin);
        return this;
    }

    /**
     * С помощью WMI в почтовом ящике {mailBoxLogin, mailBoxPwd}
     * удаляет все пользовательские папки.
     *
     * @return WmiAdapterUser
     */
    public WmiAdapterUser deleteLabels(List<String> labelNames) {
        DeleteLabelsMopsRule deleteLabelsRule = new DeleteLabelsMopsRule(auth);
        for (String labelName : labelNames) {
            Mops.deleteLabel(auth, getLid(labelName)).post(shouldBe(okSync()));
        }
        return this;
    }


    /***********
     * Возвращатели списков мидов
     **********/

    /**
     * Возвращает список из всех мидов папки Входящие, на всех страницах
     *
     * @return - Список мидов-строк
     * @throws IOException *
     */
    public List<String> getAllMidsInMailbox() throws IOException {
        List<String> allMids = new ArrayList<String>();
        for (String fid : Folders.folders(FoldersObj.empty().setUid(auth.account().uid())).get().via(auth).fids()) {
            allMids.addAll(
                    messagesByFolder(
                            MessagesByFolderObj
                                    .empty()
                                    .setUid(uid)
                                    .setFid(fid)
                                    .setFirst("0")
                                    .setCount("450")
                    )
                            .get()
                            .via(auth).mids()

            );
        }
        return allMids;
    }

    /**
     * Возвращает список из всех мидов заданой папки (currentFid), на всех страницах
     *
     * @return - Список мидов-строк
     * @throws IOException *
     */
    public List<String> getMids() throws IOException {
        return messagesByFolder(
                MessagesByFolderObj
                        .empty()
                        .setUid(uid)
                        .setFid(currentFid)
                        .setFirst("0")
                        .setCount("100")
        )
                .get()
                .via(auth).mids();

    }


    /**
     * Возвращает мид письма с заданой темой из заданой папки (currentFid), ищет только на первой странице.
     *
     * @param subject - тема письма
     * @return - Список мидов-строк
     * @throws IOException *
     */
    public String getMidOfMessageWithSubject(String subject) throws IOException {
        return getMidsOfMessageWithSubject(subject).get(0);
    }

    /**
     * Возвращает список мидов писем с заданой темой из заданой папки (currentFid), ищет только на первой странице.
     *
     * @param subject - тема письма
     * @return - Список мидов-строк
     * @throws IOException *
     */
    public List<String> getMidsOfMessageWithSubject(String subject) throws IOException {
        return messagesByFolder(
                MessagesByFolderObj
                        .empty()
                        .setUid(auth.account().uid())
                        .setFid(currentFid)
                        .setFirst("0")
                        .setCount("100")
        )
                .get()
                .via(auth).midsBySubject(subject);
    }

    /***********
     * Проверки наличия письма
     **********/

    /**
     * С помощью WMI проверяет наличие в почтовом ящике {mailBoxLogin, mailBoxPwd}
     * наличие письма с темой subject
     * <p/>
     *
     * @param subject      -- тема письма
     * @param amount       -- количество таких писем.
     * @param errorMessage -- cообщение об ошибке.
     * @param timeout      - задержка
     * @return ...
     */

    public WmiAdapterUser shouldSeeLettersWithSubject(String subject, int amount, String errorMessage, int timeout) {
        if (amount == 0) {
            int actual_amount = -1;
            try {
                Thread.sleep(5000);
                actual_amount = getMidsOfMessageWithSubject(subject).size();
            } catch (Exception e) {
                new RuntimeException("Проблемы при вытягивании списка мидов");
            }
            assertThat(errorMessage, actual_amount, equalTo(0));
        } else {
            assertThat(errorMessage,
                    auth,
                    withWaitFor(hasMsgsIn(subject, amount, currentFid), SECONDS.toMillis(timeout)));
        }
        return this;
    }

    public WmiAdapterUser shouldSeeLettersWithSubject(String subject, int amount) {
        return shouldSeeLettersWithSubject(subject,
                amount, "Письма не обнаружены у пользователя " + mailBoxLogin, DEFAULT_TIMEOUT);
    }

    public WmiAdapterUser shouldSeeLetterWithSubject(String subject, String errorMessage, int timeout) {
        return shouldSeeLettersWithSubject(subject, 1, errorMessage, timeout);
    }

    public WmiAdapterUser shouldSeeLetterWithSubject(String subject, String errorMessage) {
        return shouldSeeLetterWithSubject(subject, errorMessage, DEFAULT_TIMEOUT);
    }

    public WmiAdapterUser shouldSeeLetterWithSubject(String subject, int timeout) {
        return shouldSeeLetterWithSubject(subject, "Письмо не доставлено для " + mailBoxLogin, timeout);
    }

    public WmiAdapterUser shouldSeeLetterWithSubject(String subject) {
        return shouldSeeLetterWithSubject(subject, "Письмо не доставлено для " + mailBoxLogin);
    }

    public WmiAdapterUser shouldNotSeeLetterWithSubject(String subject, String errorMessage) {
        return shouldSeeLettersWithSubject(subject, 0, errorMessage, 0);
    }

    public WmiAdapterUser shouldNotSeeLetterWithSubject(String subject) {
        return shouldNotSeeLetterWithSubject(subject, "Письмо ошибочно доставлено для " + mailBoxLogin);
    }

    /**
     * С помощью WMI проверяет наличие в почтовом ящике {mailBoxLogin, mailBoxPwd}
     * наличие письма с темой subject и метками labelNames
     * <p/>
     *
     * @param subject    -- тема письма
     * @param labelNames -- список имён меток
     * @return WmiAdapterUser
     * @throws java.io.IOException *
     */

    public WmiAdapterUser shouldSeeLetterWithSubjectAndLabels(String subject, List<String> labelNames)
            throws IOException {
        shouldSeeLetterWithSubject(subject);
        for (String labelName : labelNames) {
            assertThat("Письмо с темой " + subject + " не пометилось меткой:" + labelNames, auth,
                    MidsWithLabelMatcher.hasMsgsWithLid(asList(getMidOfMessageWithSubject(subject)),
                            getLid(labelName)));
        }
        return this;
    }

    /**
     * С помощью WMI проверяет наличие в почтовом ящике {mailBoxLogin, mailBoxPwd}
     * наличие письма с темой subject и меткой labelName
     * <p/>
     *
     * @param subject   -- тема письма
     * @param labelName -- имя метки
     * @return WmiAdapterUser
     * @throws java.io.IOException *
     */
    public WmiAdapterUser shouldSeeLetterWithSubjectAndLabel(String subject, String labelName) throws IOException {
        shouldSeeLetterWithSubject(subject);
        //Проверяем наличие метки
        assertThat("Письмо с темой " + subject + " не пометилось меткой:" + labelName, auth,
                MidsWithLabelMatcher.hasMsgsWithLid(getMidsOfMessageWithSubject(subject), getLid(labelName)));
        return this;
    }

    public WmiAdapterUser shouldSeeLetterWithSubjectAndLabelWithLid(String subject, String lid)
            throws IOException {
        shouldSeeLetterWithSubject(subject);
        //Проверяем наличие метки
        assertThat("Письмо с темой " + subject + " не пометилось меткой:" + lid, auth,
                MidsWithLabelMatcher.hasMsgsWithLid(getMidsOfMessageWithSubject(subject), lid));
        return this;
    }

    /**
     * С помощью WMI проверяет наличие в почтовом ящике {mailBoxLogin, mailBoxPwd}
     * наличие письма с темой subject, но без метки labelName
     * <p/>
     *
     * @param subject   -- тема письма
     * @param labelName -- имя метки
     * @return WmiAdapterUser
     * @throws java.io.IOException *
     */
    public WmiAdapterUser shouldSeeLetterWithSubjectAndWithoutLabel(String subject, String labelName)
            throws IOException {
        shouldSeeLetterWithSubject(subject);
        //Проверяем наличие метки
        assertThat("Письмо с темой " + subject + " ошибочно пометилось меткой:" + labelName, auth,
                not(MidsWithLabelMatcher.hasMsgsWithLid(getMidsOfMessageWithSubject(subject), getLid(labelName))));
        return this;
    }

    public WmiAdapterUser shouldSeeLetterWithSubjectAndWithoutLabelWithLid(String subject, String lid)
            throws IOException {
        shouldSeeLetterWithSubject(subject);
        //Проверяем отсутствие метки
        assertThat("Письмо с темой " + subject + " ошибочно пометилось меткой:" + lid, auth,
                not(MidsWithLabelMatcher.hasMsgsWithLid(getMidsOfMessageWithSubject(subject), lid)));
        return this;
    }

    /**
     * С помощью WMI проверяет в почтовом ящике {mailBoxLogin, mailBoxPwd}
     * наличие письма с темой subject и проверят есть ли отправитель expectedToHeaderValue у этого письма.
     *
     * @param subject               -- тема письма
     * @param expectedToHeaderValue -- отправитель
     * @return WmiAdapterUser
     * @throws java.io.IOException *
     */
    public WmiAdapterUser shouldSeeMessageWithSubjectAndToHeader(String subject, String expectedToHeaderValue)
            throws IOException {
        shouldSeeLetterWithSubject(subject);
        assertThat("Письмо с темой'" + subject + "' не имеет в поле TO нужного получателя.",
                FilterSearchCommand
                        .filterSearch(FilterSearchObj.empty().setUid(getUid())
                                .setMids(getMidOfMessageWithSubject(subject)))
                        .get()
                        .via(auth)
                        .to()
                        .stream()
                        .map(email -> String.format("%s@%s", email.get("local"), email.get("domain")).toLowerCase())
                        .collect(Collectors.toList()), hasItem(expectedToHeaderValue));
        return this;
    }

    public WmiAdapterUser shouldSeeMessageWithSubjectAndFromInMailList(String subject, String expectedFromValue)
            throws IOException {
        shouldSeeLetterWithSubject(subject);
        Map<String, String> from = FilterSearchCommand
                .filterSearch(FilterSearchObj.empty().setUid(getUid())
                        .setMids(getMidOfMessageWithSubject(subject)))
                .get()
                .via(auth)
                .from();
        assertThat("Письмо с темой'" + subject + "' не имеет в поле From нужного отправителя.",
                from.get("local") + "@" + from.get("domain"), equalTo(expectedFromValue));
        return this;
    }

    /**
     * ********
     * Для проверок содержимого письма
     * ********
     */
    public WmiAdapterUser shouldSeeLetterWithSubjectAndContent(String subject, Matcher<String> contentMatcher)
            throws IOException, MessagingException {
        shouldSeeLetterWithSubject(subject);
        TestMessage receivedMsg = getMessageWithSubject(subject);
        assertThat("Содержимое полученного письма неверно!",
                receivedMsg, hasContent(contentMatcher));
        return this;
    }

    public TestMessage getMessageWithMid(String mid) throws IOException, MessagingException {
        String source = MbodyApi.apiMbody(auth.account().userTicket()).messageSource()
                .withUid(uid)
                .withMid(mid)
                .get(identity()).peek().asString();

        Gson gson = new GsonBuilder().create();
        JSONObject respJson = null;
        String messageSource = "";
        try {
            respJson = new JSONObject(source);
            assertThat("Проверяем наличие поля text в полученном ответе", respJson.has("text"));
            messageSource = decodeBase64(respJson.getString("text"));
        } catch (JSONException e) {
            new RuntimeException("Проблемы при вычитывании тела письма");
        }

        String charset = "UTF-8";
        if (messageSource.contains("charset=windows-1251")) {
            charset = "windows-1251";
        } else if (messageSource.contains("charset=koi8-r")) {
            charset = "koi8-r";
        }
        return new TestMessage(new MimeMessage(getDefaultInstance(new Properties()),
                toInputStream(messageSource, charset)));
    }

    public TestMessage getMessageWithSubject(String subject) throws IOException, MessagingException {
        return getMessageWithMid(getMidOfMessageWithSubject(subject));
    }

    public static String decodeBase64(String source) {
        byte[] decodedSource = Base64.getDecoder().decode(source);
        return new String(decodedSource);
    }
}