package ru.yandex.autotests.innerpochta.wmi.core.base;

import org.apache.commons.lang.RandomStringUtils;
import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import ru.yandex.autotests.innerpochta.wmi.core.exceptions.RetryException;
import ru.yandex.autotests.innerpochta.wmicommon.Util;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.*;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import static java.lang.String.valueOf;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.apache.commons.lang.StringUtils.defaultIfEmpty;
import static ru.yandex.autotests.plugins.testpers.html.common.HtmlUtils.accordeon;
import static ru.yandex.autotests.plugins.testpers.html.common.HtmlUtils.html;

/**
 * Created with IntelliJ IDEA.
 * User: lanwen
 * Date: 30.12.12
 * Time: 23:17
 */
public class DocumentConverter implements XpathConverter {

    private String response;

    private Document document;

    protected XPathFactory factory = XPathFactory.newInstance();

    protected XPath xpath = factory.newXPath();

    private XPathExpression xPathExpression;

    private boolean needLog = false;

    private Logger logger;

    private String logMessage;


    public static XpathConverter from(String response) {
        return new DocumentConverter().read(response);
    }


    public static XpathConverter from(Document response) {
        return new DocumentConverter().read(response);
    }


    @Override
    public XpathConverter read(String response) {
        this.response = response;
        document = null;
        return this;
    }


    @Override
    public XpathConverter read(Document response) {
        this.response = null;
        document = response;
        return this;
    }

    @Override
    public Document getConverted() {
        if (document != null) {
            return document;
        }
        try {

            DocumentBuilderFactory domfactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder;

            builder = domfactory.newDocumentBuilder();
            document = builder.parse(new InputSource(new StringReader(response)));

            return document;
        } catch (Exception e) {
            String replace = RandomStringUtils.randomAlphanumeric(20); // To avoid escaping twice
            throw new RetryException(
                    html(accordeon("Ошибка при парсинге xml", replace)).replace(replace, response), e);
        }
    }


    @Override
    public DocumentConverter byXpath(String xpathString) {
        try {
            xPathExpression = xpath.compile(xpathString);
            logMessage = defaultIfEmpty(logMessage, xpathString);
        } catch (XPathExpressionException e) {
            throw new RuntimeException("XPath выражение " + xpathString + " некорректно", e);
        }
        return this;
    }


    @Override
    public XpathConverter log(Logger logger, String logMessage) {
        this.logger = logger;
        this.logMessage = logMessage;
        this.needLog = true;
        return this;
    }

    @Override
    public XpathConverter log(Logger logger) {
        return log(logger, null);
    }

    public String asString() {
        try {
            String result = xPathExpression.evaluate(getConverted());
            logIfNeed(result);
            return result;
        } catch (XPathExpressionException e) {
            throw new RuntimeException("Невозможно вернуть как строку данный xpath", e);
        }
    }


    public String asXmlString() {
        NodeList list = asNodeList();
        StringWriter sw = new StringWriter();
        Transformer serializer = null;
        try {
            serializer = TransformerFactory.newInstance().newTransformer();
            serializer.transform(new DOMSource(list.item(0)), new StreamResult(sw));
            return sw.toString();
        } catch (TransformerException e) {
            throw new RuntimeException("Невозможно получить отконвертированный xml в строку", e);
        }
    }

    public Integer asInteger() {
        try {
            Integer integer = ((Double) xPathExpression.evaluate(getConverted(), XPathConstants.NUMBER)).intValue();
            logIfNeed(integer);
            return integer;
        } catch (XPathExpressionException e) {
            throw new RuntimeException("Невозможно вернуть как Number данный xpath " + logMessage, e);
        }
    }

    public NodeList asNodeList() {
        try {
            return (NodeList) xPathExpression.evaluate(getConverted(), XPathConstants.NODESET);
        } catch (XPathExpressionException e) {
            throw new RuntimeException("Невозможно вернуть NodeList", e);
        }
    }

    public List<String> asList() {
        return Util.nodesToArrayList(asNodeList());
    }

    public Boolean asBoolean() {
        try {
            Boolean evaluated = (Boolean) xPathExpression.evaluate(getConverted(), XPathConstants.BOOLEAN);
            logIfNeed(evaluated);
            return evaluated;
        } catch (XPathExpressionException e) {
            throw new RuntimeException("Невозможно вернуть Boolean", e);
        }
    }

    public String asSeconds() {
        try {
            String sec = valueOf(MILLISECONDS.toSeconds(Long.parseLong(xPathExpression.evaluate(getConverted()))));
            logIfNeed(sec);
            return sec;
        } catch (XPathExpressionException e) {
            throw new RuntimeException("Невозможно вернуть в секундах данный xpath", e);
        }
    }

    public <T> T as(Class<T> clazz) {
        try {
            Node node = (Node) xPathExpression.evaluate(getConverted(), XPathConstants.NODE);
            return unmarshallNodeTo(node, clazz);
        } catch (XPathExpressionException e) {
            throw new RuntimeException("Проблема в xpath", e);
        }
    }

    public <T> List<T> asListOf(Class<T> clazz) {
        NodeList nodes = asNodeList();
        List<T> entries = new ArrayList<T>();

        for (int i = 0; i < nodes.getLength(); i++) {
            Node node = nodes.item(i);
            entries.add(unmarshallNodeTo(node, clazz));
        }
        return entries;
    }

    private void logIfNeed(Object evaluated) {
        if (needLog) {
            logger.info(logMessage + ": " + evaluated);
        }
        logger = null;
        logMessage = null;
        needLog = false;
    }


    private <T> T unmarshallNodeTo(Node node, Class<T> clazz) {
        try {
            JAXBContext context = JAXBContext.newInstance(clazz);
            Unmarshaller unmarshaller = context.createUnmarshaller();
            return unmarshaller.unmarshal(node, clazz).getValue();
        } catch (JAXBException e) {
            throw new RuntimeException("Проблема при демаршаллизации в " + clazz.getCanonicalName(), e);
        }
    }

}