package ru.yandex.autotests.innerpochta.wmi.core.filter;

import com.google.common.base.Charsets;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.fluent.Request;
import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeMatcher;
import ru.yandex.autotests.innerpochta.wmi.core.base.Context;
import ru.yandex.autotests.innerpochta.wmi.core.matchers.IsArrayContaining;

import java.io.IOException;
import java.util.stream.Stream;

import static java.lang.String.format;
import static java.util.stream.Collectors.joining;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.not;

/**
 * Created with IntelliJ IDEA.
 * User: lanwen
 * Date: 01.01.13
 * Time: 19:22
 */
public class CheckHeaderFilter implements Filter {
    Matcher<Header[]> headerMatcher;

    private CheckHeaderFilter(Matcher<Header[]> headerMatcher) {
        this.headerMatcher = headerMatcher;
    }

    public static CheckHeaderFilter hasHeader(String headerName, Matcher<String> matcher) {
        Matcher<Header[]> headMatcher = headerInArray(asHeader(headerName, matcher));
        return new CheckHeaderFilter(headMatcher);
    }


    public static CheckHeaderFilter hasHeader(String headerName) {
        Matcher<Header[]> hm = headerInArray(asHeader(headerName, not("")));
        return new CheckHeaderFilter(hm);
    }

    public static CheckHeaderFilter hasNoHeader(String headerName) {
        Matcher<Header[]> hm = headerInArray(asHeader(headerName, not("")));
        return new CheckHeaderFilter(not(hm));
    }

    @Override
    public Request filter(Request request, Context ctx) {
        return ctx.next(request);
    }

    @Override
    public HttpResponse filter(HttpResponse response, Context ctx) throws IOException {
        assertThat(response.getAllHeaders(), headerMatcher);
        return ctx.next(response);
    }

    private static Matcher<Header> asHeader(final String headerName, final Matcher<String> matcher) {
        return new TypeSafeMatcher<Header>() {
            @Override
            protected boolean matchesSafely(Header item) {
                return item.getName().equals(headerName) && matcher.matches(iso8859toUTF8Value(item));
            }

            @Override
            public void describeTo(Description description) {
                description.appendText("Header ").appendValue(headerName)
                        .appendText(" with value ").appendDescriptionOf(matcher);
            }

            @Override
            protected void describeMismatchSafely(Header item, Description mismatchDescription) {
                mismatchDescription.appendText("was headers:\n").appendText(printHeaders(item));
            }
        };
    }

    public static Matcher<Header[]> headerInArray(final Matcher<Header> matcher) {
        return new IsArrayContaining<Header>(matcher) {
            @Override
            public void describeMismatchSafely(Header[] item, Description mismatchDescription) {
                mismatchDescription.appendText("was headers:\n").appendText(printHeaders(item));
            }
        };
    }

    public static String printHeaders(Header... headers) {
        return Stream.of(headers)
                .map(header -> format("\t- %s: %s", header.getName(), iso8859toUTF8Value(header)))
                .collect(joining("\n"));
    }

    public static String iso8859toUTF8Value(Header header) {
        return new String(header.getValue().getBytes(Charsets.ISO_8859_1), Charsets.UTF_8);
    }
}
