package ru.yandex.autotests.innerpochta.wmi.core.filter.log;

import com.google.common.base.Charsets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.fluent.Request;
import org.apache.log4j.Logger;
import org.rendersnake.HtmlCanvas;
import ru.yandex.autotests.innerpochta.wmi.core.base.Context;
import ru.yandex.autotests.innerpochta.wmi.core.oper.Oper;
import ru.yandex.autotests.plugins.testpers.html.common.Code;
import ru.yandex.qatools.allure.annotations.Attachment;
import ru.yandex.qatools.allure.annotations.Step;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import static ru.yandex.autotests.innerpochta.wmi.core.Common.entityToString;
import static ru.yandex.autotests.plugins.testpers.html.common.Code.codeBlock;
import static ru.yandex.autotests.plugins.testpers.html.common.ResponseRender.blocks;
import static ru.yandex.qatools.elliptics.ElClient.elliptics;

/**
 * Created with IntelliJ IDEA.
 * User: lanwen
 * Date: 27.12.12
 * Time: 18:36
 */
public class BootstrapLoggingFilter extends LoggingFilter {
    // IDEA configLog.writeToInfo().conditionalAppend(…).conditionalAppend…

    private RequestHeaderInterceptor headerInterceptor;


    @Override
    public Request filter(Request request, Context ctx) {
        headerInterceptor = new RequestHeaderInterceptor();
        ctx.client().addRequestInterceptor(headerInterceptor);

        return ctx.next(request);
    }

    @Override
    public HttpResponse filter(HttpResponse response, Context ctx) throws IOException {
        Logger logger = Logger.getLogger(ctx.resp());
        if (onlyIfError && logCondition.matches(response)) {
            logInAllure(ctx, response);
            logDescription(ctx, logger);
            logUrl(ctx, logger, response);
            logPostBody(ctx, logger);
        }

        if (!onlyIfError) {
            logInAllure(ctx, response);
            logDescription(ctx, logger);
            logUrl(ctx, logger, response);
            logPostBody(ctx, logger);
        }

        return ctx.next(response);
    }

    private void logInAllure(Context ctx, HttpResponse response) throws IOException {
        StringBuilder sb = new StringBuilder();
        sb.append(appendIf(logUrl, ctx.requestType().name() + ": " + ctx.url()));
        sb.append(appendIf(logParams, ctx.params().asGet(!ctx.cmd().contains("?"))));
        if (!isEmpty(sb.toString())) {
            logUrlInAllure(sb.toString(), ctx, response);
        }
    }

    @Step("{0}")
    private void logUrlInAllure(String url, Context ctx, HttpResponse response) throws IOException {
        List<Code> blocks = new ArrayList<>();

        //REQ:

        if (logDescription && isNotEmpty(ctx.descr())) {
            blocks.add(codeBlock("REQUEST DESCRIPTION", ctx.descr()));
        }
        
        if (logUrl && isNotEmpty(url)) {
            blocks.add(codeBlock("REQUEST", url));
        }

        if (logReqHeaders && isNotEmpty(headersToStringAllure(headerInterceptor.getHeaders()))) {
            blocks.add(codeBlock("REQUEST HEADERS", headersToStringAllure(headerInterceptor.getHeaders())));
        }

        if (logPostBody && ctx.params().hasBody() && isNotEmpty(ctx.params().getBody())) {
            blocks.add(codeBlock("REQUEST POST BODY", ctx.params().getBody()));
        }

        //RESP:

        if (logStatusLine && isNotEmpty(response.getStatusLine().toString())) {
            blocks.add(codeBlock("RESPONSE STATUS", response.getStatusLine().toString()));
        }

        if (logRespHeaders && isNotEmpty(headersToStringAllure(response.getAllHeaders()))) {
            blocks.add(codeBlock("RESPONSE HEADERS", headersToStringAllure(response.getAllHeaders())));
        }

        if (logRespBody && isNotEmpty(respLogging(response))) {
            blocks.add(codeBlock("RESPONSE", entityToString(response.getEntity(), Charsets.UTF_8)));
        }

        if(CollectionUtils.isNotEmpty(blocks)) {
            blocks.stream().forEach(block -> block.needRender(true));
            attachSmth(blocks);
        }
    }

    @Attachment("REQ & RESP")
    private String attachSmth(List<Code> blocks) throws IOException {
        return new HtmlCanvas().render(blocks(blocks)).toHtml();
    }

    private void logPostBody(Context ctx, Logger logger) {
        if (logPostBody && ctx.params().hasBody()) {
            logger.info("[[a href=" +
                    logStringResponseToFileAndReturnPath(ctx.resp(), ctx.params().getBody())
                    + "]]POST BODY[[/a]]"
            );
        }
    }

    private void logUrl(Context ctx, Logger logger, HttpResponse response) throws IOException {
        StringBuilder sb = new StringBuilder();

        sb.append(appendIf(logRespBody || logRespHeaders || logStatusLine || logReqHeaders,
                "[[div class='btn btn-mini btn-info' data-placement='bottom' "));

        sb.append(appendIf(logStatusLine, " title='" + getStatus(response) + "'"));

        sb.append(appendIf(logRespHeaders || logRespBody || logReqHeaders, " data-content='"));

        sb.append(appendIf(logRespBody, "[[a href=" +
                logStringResponseToFileAndReturnPath(ctx.resp(), respLogging(response)) +
                "]]RESP BODY[[/a]][[br/]][[br/]]"));

        sb.append(appendIf(logReqHeaders, "[[b]]REQUEST HEADERS:[[/b]] [[br/]]" +
                headersToString(headerInterceptor.getHeaders()) + "[[br/]]"));

        sb.append(appendIf(logRespHeaders, "[[b]]RESP HEADERS:[[/b]] [[br/]]" +
                headersToString(response.getAllHeaders())));


        sb.append(appendIf(logRespHeaders || logRespBody || logReqHeaders, "'"));

        sb.append(appendIf(logRespBody || logRespHeaders || logStatusLine || logReqHeaders, "]]"));

        sb.append(appendIf(logUrl || logRespBody || logRespHeaders || logStatusLine || logReqHeaders,
                ctx.requestType().name()));

        sb.append(appendIf(logRespBody || logRespHeaders || logStatusLine || logReqHeaders, "[[/div]]"));

        sb.append(appendIf(logUrl, " " + ctx.url()));

        sb.append(appendIf(logParams, ctx.params().asGet(!ctx.cmd().contains("?"))));

        logOnlyNotEmpty(logger, sb.toString());
    }


    private void logDescription(Context ctx, Logger logger) {
        logOnlyNotEmpty(logger, appendIf(logDescription, ctx.descr()));
    }

    public String respLogging(HttpResponse response) throws IOException {
        return new HtmlCanvas()
                .render(blocks(
                        codeBlock("RESP: ", entityToString(response.getEntity(), Charsets.UTF_8))
                                .needRender(logRespBody)
                )).toHtml();
    }

    private String getStatus(HttpResponse response) {
        return "STATUS: " + response.getStatusLine().toString();
    }


    /**
     * Логирование ответа в файл
     */
    protected <T extends Oper> String logStringResponseToFileAndReturnPath(Class<T> clazz, String resp) {
        if (isEmpty(resp)) {
            return "";
        }
        return elliptics().path(clazz).randomize().name(clazz.getSimpleName() + "_.htm").put(resp).get().url();
    }


    /**
     * Принимает массив хэдеров и возвращает строку
     *
     * @param headers -  массив хедеров
     * @return - строка из хедеров в столбик
     */
    public static String headersToString(Header[] headers) {
        return formatHeaders(headers, "%s: %s;[[br/]]");
    }

    public static String headersToStringAllure(Header[] headers) {
        return formatHeaders(headers, "%s: %s;");
    }

    private String appendIf(Boolean condition, String toLog) {
        if (condition) {
            return toLog;
        }
        return "";
    }

    private void logOnlyNotEmpty(Logger logger, String msg) {
        if (!isEmpty(msg)) {
            logger.info(msg);
        }
    }


}

