package ru.yandex.autotests.innerpochta.wmi.core.filter.log;

import org.apache.http.HttpResponse;
import org.apache.http.util.EntityUtils;
import org.hamcrest.*;
import ru.yandex.autotests.innerpochta.wmi.core.filter.Filter;

import java.io.IOException;

import static org.hamcrest.Matchers.*;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;

/**
 * Created with IntelliJ IDEA.
 * User: lanwen
 * Date: 28.12.12
 * Time: 16:45
 */
public class LoggerFilterBuilder {

    private LoggingFilter logger;

    private LoggerFilterBuilder() {
        if (props().logVersion().equals("local")) {
            logger = new LoggingFilter();
        } else {
            logger = new BootstrapLoggingFilter();
        }
    }

    public LoggerFilterBuilder descr(Boolean descr) {
        logger.setLogDescription(descr);
        return this;
    }


    public LoggerFilterBuilder body(Boolean body) {
        logger.setLogRespBody(body);
        return this;
    }


    public LoggerFilterBuilder respHeaders(Boolean value) {
        logger.setLogRespHeaders(value);
        return this;
    }

    public LoggerFilterBuilder reqHeaders(Boolean value) {
        logger.setLogReqHeaders(value);
        return this;
    }


    public LoggerFilterBuilder url(boolean value) {
        logger.setLogUrl(value);
        return this;
    }

    public LoggerFilterBuilder postBody(boolean value) {
        logger.setLogPostBody(value);
        return this;
    }

    public LoggerFilterBuilder params(boolean value) {
        logger.setLogParams(value);
        return this;
    }

    public LoggerFilterBuilder cooks(boolean value) {
        logger.setLoReqCookies(value);
        return this;
    }


    public LoggerFilterBuilder status(Boolean value) {
        logger.setLogStatusLine(value);
        return this;
    }


    public LoggerFilterBuilder onlyIfCondition(Matcher<HttpResponse> condition) {
        logger.setOnlyIfError(true);
        logger.setLogCondition(condition);
        return this;
    }

    public LoggerFilterBuilder nothing() {
        return descr(false)
                .body(false)
                .reqHeaders(false)
                .respHeaders(false)
                .status(false)
                .url(false)
                .postBody(false)
                .cooks(false)
                .params(false);
    }

    public LoggerFilterBuilder onlyIfError() {
        return onlyIfCondition(statusCode(greaterThanOrEqualTo(400)));
    }

    public LoggerFilterBuilder onlyIfErrorOrXmlWithError() {
        return onlyIfCondition(anyOf(statusCode(greaterThanOrEqualTo(400)), response(containsString("error"))));
    }

    public Filter build() {
        return logger;
    }


    @Factory
    public static LoggerFilterBuilder log() {
        return new LoggerFilterBuilder();
    }


    public static Matcher<HttpResponse> statusCode(final Matcher<Integer> matcher) {
        return new TypeSafeMatcher<HttpResponse>() {

            @Override
            protected boolean matchesSafely(HttpResponse item) {
                return matcher.matches(item.getStatusLine().getStatusCode());
            }

            @Override
            public void describeTo(Description description) {
            }
        };
    }

    public static Matcher<HttpResponse> response(final Matcher<String> matcher) {
        return new FeatureMatcher<HttpResponse, String>(matcher, "тело содержит", "содержимое тело") {
            @Override
            protected String featureValueOf(HttpResponse actual) {
                try {
                    return EntityUtils.toString(actual.getEntity(), "UTF-8");
                } catch (IOException e) {
                    throw new RuntimeException("", e);
                }
            }
        };
    }

}
