package ru.yandex.autotests.innerpochta.wmi.core.filter.log;

import ch.lambdaj.function.convert.Converter;
import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import org.apache.http.*;
import org.apache.http.client.fluent.Request;
import org.apache.http.protocol.HttpContext;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.hamcrest.Matcher;
import org.rendersnake.HtmlCanvas;
import ru.yandex.autotests.innerpochta.util.LogToFileUtils;
import ru.yandex.autotests.innerpochta.wmi.core.base.Context;
import ru.yandex.autotests.innerpochta.wmi.core.filter.Filter;
import ru.yandex.autotests.innerpochta.wmi.core.oper.Oper;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static ch.lambdaj.Lambda.convert;
import static java.lang.String.format;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static ru.yandex.autotests.innerpochta.wmi.core.Common.entityToString;
import static ru.yandex.autotests.plugins.testpers.html.common.Code.codeBlock;
import static ru.yandex.autotests.plugins.testpers.html.common.ResponseRender.blocks;

/**
 * Created with IntelliJ IDEA.
 * User: lanwen
 * Date: 27.12.12
 * Time: 18:36
 */
public class LoggingFilter implements Filter {
    // TODO Собственное имя логгера
    // IDEA configLog.writeToInfo().conditionalAppend(…).conditionalAppend…

    protected boolean logDescription = true;
    protected boolean logUrl = true;
    protected boolean logPostBody = true;
    protected boolean logParams = true;
    protected boolean logReqHeaders = true;
    protected boolean loReqCookies = false;

    protected boolean logStatusLine = true;
    protected boolean logRespBody = true;
    protected boolean logRespHeaders = true;
    protected boolean logRespCookies = false;

    protected boolean onlyIfError = false;
    protected Matcher<HttpResponse> logCondition;


    public void setLogDescription(boolean logDescription) {
        this.logDescription = logDescription;
    }

    public void setLogUrl(boolean logUrl) {
        this.logUrl = logUrl;
    }

    public void setLogPostBody(boolean logPostBody) {
        this.logPostBody = logPostBody;
    }

    public void setLogParams(boolean logParams) {
        this.logParams = logParams;
    }

    public void setLogReqHeaders(boolean logReqHeaders) {
        this.logReqHeaders = logReqHeaders;
    }

    public void setLoReqCookies(boolean loReqCookies) {
        this.loReqCookies = loReqCookies;
    }

    public void setLogStatusLine(boolean logStatusLine) {
        this.logStatusLine = logStatusLine;
    }

    public void setLogRespBody(boolean logRespBody) {
        this.logRespBody = logRespBody;
    }

    public void setLogRespHeaders(boolean logRespHeaders) {
        this.logRespHeaders = logRespHeaders;
    }

    public void setLogRespCookies(boolean logRespCookies) {
        this.logRespCookies = logRespCookies;
    }

    public void setOnlyIfError(boolean onlyIfError) {
        this.onlyIfError = onlyIfError;
    }

    public void setLogCondition(Matcher<HttpResponse> logCondition) {
        this.logCondition = logCondition;
    }

    @Override
    public Request filter(Request request, Context ctx) {
        Logger logger = Logger.getLogger(ctx.resp());
        if (!onlyIfError) {
            logDescription(ctx, logger);
            logUrl(ctx, logger);
            logPostBody(ctx);
        }
        return ctx.next(request);
    }

    @Override
    public HttpResponse filter(HttpResponse response, Context ctx) throws IOException {
        Logger logger = Logger.getLogger(ctx.resp());
        if (onlyIfError && logCondition.matches(response)) {
            logDescription(ctx, logger);
            logUrl(ctx, logger);
            logPostBody(ctx);

            logStringResponseToFile(ctx.resp(), respLogging(response));
        }

        logStringResponseToFile(ctx.resp(), appendIf(!onlyIfError, respLogging(response)));

        return ctx.next(response);
    }

    private void logPostBody(Context ctx) {
        logStringResponseToFile(ctx.resp(),
                appendIf(
                        logPostBody && ctx.params().hasBody(),
                        ctx.params().getBody()
                ));
    }

    private void logUrl(Context ctx, Logger logger) {
        StringBuilder sb = new StringBuilder();

        sb.append(appendIf(logUrl, ctx.requestType().name() + ": " + ctx.url()));

        sb.append(appendIf(logParams, ctx.params().asGet(!ctx.cmd().contains("?"))));

        logOnlyNotEmpty(logger, sb.toString());
    }


    private void logDescription(Context ctx, Logger logger) {
        logOnlyNotEmpty(logger, appendIf(logDescription, ctx.descr()));
    }

    public String respLogging(HttpResponse response) throws IOException {
        return new HtmlCanvas().render(blocks(
                codeBlock("STATUS-LINE: ", response.getStatusLine().toString()).needRender(logStatusLine),
                codeBlock("RESP HEADERS: ", headersToString(response.getAllHeaders())).needRender(logRespHeaders),
                codeBlock("RESP: ", entityToString(response.getEntity(), Charsets.UTF_8)).needRender(logRespBody)
        )).toHtml();
    }


    /**
     * Логирование ответа в файл
     */
    protected <T extends Oper> void logStringResponseToFile(Class<T> clazz, String resp) {
        try {
            if (isEmpty(resp)) {
                return;
            }
            File htm = LogToFileUtils.getLogFileWithoutLogEvent(clazz.getSimpleName() + "_", "htm");
            LogToFileUtils.writeToFile(htm, resp);
            LogManager.getLogger(clazz).info(format("Resp: [ %s ]", htm.getAbsolutePath()));
        } catch (NullPointerException e) {
            throw new RuntimeException("Похоже не указана пропертя -Dtest.report.dir=reports", e);
        } catch (Exception e) {
            throw new RuntimeException("Ошибка при логировании ответа в файл", e);
        }
    }


    private static String headersToString(Header[] headers) {
        return formatHeaders(headers, "%s: %s;");
    }

    public static String formatHeaders(Header[] headers, final String pattern) {
        return Joiner.on("\n").join(convert(headers, new Converter<Header, String>() {
            @Override
            public String convert(Header from) {
                return format(pattern, from.getName(), from.getValue());
            }
        }));
    }


    private String appendIf(Boolean condition, String toLog) {
        if (condition) {
            return toLog;
        }
        return "";
    }

    private void logOnlyNotEmpty(Logger logger, String msg) {
        if (msg != null && !msg.isEmpty()) {
            logger.info(msg);
        }
    }

    public static class RequestHeaderInterceptor implements HttpRequestInterceptor {
        private Header[] headers = new Header[]{};

        public Header[] getHeaders() {
            headers = removeHeaderFrom(headers, "Cookie");
            headers = removeHeaderFrom(headers, "Content-Type");
            return headers;
        }

        private Header[] removeHeaderFrom(Header[] headers, String headerName) {
            List<Header> headerList = new ArrayList<Header>();
            for (Header header : headers) {
                if (!header.getName().equals(headerName)) {
                    headerList.add(header);
                }
            }
            return headerList.toArray(new Header[headerList.size()]);
        }

        @Override
        public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {
            headers = request.getAllHeaders();
        }
    }

}

