package ru.yandex.autotests.innerpochta.wmi.core.matchers;

import difflib.DiffUtils;
import difflib.Patch;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.LogManager;
import org.custommonkey.xmlunit.DetailedDiff;
import org.custommonkey.xmlunit.Difference;
import org.custommonkey.xmlunit.NodeDescriptor;
import org.custommonkey.xmlunit.NodeDetail;
import org.custommonkey.xmlunit.XMLConstants;
import org.custommonkey.xmlunit.XMLUnit;
import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.TypeSafeMatcher;
import org.junit.ComparisonFailure;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;
import ru.yandex.autotests.innerpochta.wmicommon.Util;
import ru.yandex.qatools.allure.annotations.Attachment;
import ru.yandex.qatools.elliptics.ElGet;
import ru.yandex.qatools.pagediffer.PageDiffer;
import ru.yandex.qatools.pagediffer.config.PageDifferConfig;
import ru.yandex.qatools.pagediffer.diff.Diff;
import ru.yandex.qatools.pagediffer.diff.DocumentPatch;
import ru.yandex.qatools.pagediffer.diff.report.StandardReportGenerator;
import ru.yandex.qatools.pagediffer.document.XmlDocument;
import ru.yandex.qatools.pagediffer.document.filter.DocumentFilter;
import ru.yandex.qatools.pagediffer.document.filter.ExcludeXmlNodesFilter;
import ru.yandex.qatools.pagediffer.exceptions.PageDifferException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static java.lang.String.format;
import static java.util.Arrays.asList;
import static org.apache.commons.lang3.StringUtils.abbreviate;
import static ru.yandex.autotests.innerpochta.wmicommon.WmiConsts.DEBUG_NODE;
import static ru.yandex.qatools.elliptics.ElClient.elliptics;

/**
 * Created with IntelliJ IDEA.
 * User: lanwen
 * Date: 30.01.13
 * Time: 20:24
 */
public class DocumentCompareMatcher extends TypeSafeMatcher<org.w3c.dom.Document> {
    private String urlcomment = "";
    
    private String path = "";
    
    private XmlDocument original;
    private XmlDocument modified;
    private DocumentPatch<XmlDocument> patch;

    public DocumentCompareMatcher(XmlDocument original) {
        this.original = original;
    }

    private List<DocumentFilter<XmlDocument>> filters = new ArrayList<>();

    @Override
    protected boolean matchesSafely(Document newDoc) {
        PageDifferConfig.getConfig().setTextDiffType(PageDifferConfig.TextDiffType.WORD_BY_WORD);
        modified = new XmlDocument(Util.convertDocumentToString(newDoc));
        try {
            applyFiltersTo(modified, original);

            patch = patch(original, modified);
            return patch.getDeltas().isEmpty();
        } catch (PageDifferException e) {
            throw new RuntimeException("Ошибка диффера", e);
        }
    }

    public DocumentCompareMatcher urlcomment(String urlcomment) {
        this.urlcomment = urlcomment;
        return this;
    }

    public DocumentCompareMatcher filterWithCondition(DocumentFilter<XmlDocument> filter, boolean toFilter) {
        if (toFilter) {
            filters.add(filter);
        }
        return this;
    }

    public DocumentCompareMatcher exclude(String xpath) {
        if (StringUtils.isEmpty(xpath)) {
            return this;
        }
        filters.add(new ExcludeXmlNodesFilter<>(xpath));
        return this;
    }

    public DocumentCompareMatcher excludeAll(String... xpaths) {
        return excludeAll(asList(xpaths));
    }

    public DocumentCompareMatcher excludeAll(List<String> xpaths) {
        for (String xpath : xpaths) {
            exclude(xpath);
        }
        return this;
    }

    private void applyFiltersTo(XmlDocument... docs) throws PageDifferException {
        for (XmlDocument document : docs) {
            for (DocumentFilter<XmlDocument> filter : filters) {
                filter.apply(document);
            }
        }
    }

    /**
     * Available only after describeMismatch
     * @return elliptics path
     */
    public String getSavedPath() {
        return path;
    }

    @Override
    public void describeTo(Description description) {
        description.appendText("Не должно быть различий");
    }

    @Override
    protected void describeMismatchSafely(Document item, Description mismatchDescription) {
        Diff<XmlDocument> diff = PageDiffer.diff(original, modified);
        mismatchDescription.appendText("Различий ").appendValue(diff.getDifferencesNumber()).appendText("\n");
        try {
            ElGet diffReport = elliptics()
                    .path(DocumentCompareMatcher.class)
                    .randomize().name("_pagediffer_" + urlcomment + ".htm")
                    .put(new StandardReportGenerator<>().generateReport(diff).toString())
                    .log(LogManager.getLogger("DIFFReport")).pattern("[[a href={url}]]DiffReport[[/a]]")
                    .get();
            path = diffReport.fullpath();
            String url = diffReport.url();

            attachDiff(diffReport.asString());
            String logpath = "[[a target='_blank' href=" + url + " class='pagediffer']]DiffReport[[/a]]";

            mismatchDescription
                    .appendText(logpath)
                    .appendValueList("\n-> ", "\n-> ", "", NodeDescr.getAllDiffs(diff));
        } catch (Exception e) {
            throw new RuntimeException("Проблема сохранения лога отличий", e);
        }
    }

    @Attachment("DIFF REPORT")
    private String attachDiff(String diffReport) {
        return diffReport;
    }


    @Factory
    public static DocumentCompareMatcher equalToDoc(Document baseDoc) {
        return new DocumentCompareMatcher(new XmlDocument(Util.convertDocumentToString(baseDoc)))
                .exclude("//timer_mulca")
                .exclude("//timer_db")
                .exclude("//timer_logic")
                .exclude("//timer_search")
                .exclude("//scn")
                .exclude("//ckey")
                .exclude("//password_verification_age")
                        //прилетает из бб-рц
                .exclude("//bb_connection_id")
                .exclude("//yamail/account_information/birthday")
                        //нужно для тестирования сброса кэша
                .exclude("//" + DEBUG_NODE);
    }


    private static class NodeDescr extends NodeDescriptor {
        public static List<String> getAllDiffs(Diff<XmlDocument> differDiff) throws IOException, SAXException {

            XMLUnit.setIgnoreWhitespace(true);
            List<String> diffs = new ArrayList<>();
            DetailedDiff diff = new DetailedDiff(XMLUnit.compareXML(
                    differDiff.getOriginalDocument().toString(),
                    differDiff.getModifiedDocument().toString()
            ));
            for (Object difference : diff.getAllDifferences()) {
                Difference df = (Difference) difference;

                if (df.getControlNodeDetail() == null || df.getTestNodeDetail() == null) {
                    diffs.add(df.getDescription());
                    continue;
                }

                String format = format("%s - %s (%s)",
                        df.getDescription(),
                        abbreviate(descr(df.getControlNodeDetail()), 1000),
                        abbreviate(new ComparisonFailure("",
                                df.getControlNodeDetail().getValue(),
                                df.getTestNodeDetail().getValue()
                        ).getMessage(), 1000));
                diffs.add(format);
            }
            return diffs;
        }

        /**
         * Копипаста для оверрайда. И чего бы им не сделать этот метод протектедом?
         *
         * @param nodeDetail
         *
         * @return
         */
        public static String descr(NodeDetail nodeDetail) {
            Node aNode = nodeDetail.getNode();
            if (aNode == null) {
                return nodeDetail.getXpathLocation();
            }
            StringBuffer buf = new StringBuffer(XMLConstants.OPEN_START_NODE);
            switch (aNode.getNodeType()) {
                case Node.ATTRIBUTE_NODE:
                    appendAttributeDetail(buf, aNode);
                    break;
                case Node.ELEMENT_NODE:
                    appendElementDetail(buf, aNode, true);
                    break;
                case Node.TEXT_NODE:
                    appendTextDetail(buf, aNode);
                    break;
                case Node.CDATA_SECTION_NODE:
                    appendCdataSectionDetail(buf, aNode);
                    break;
                case Node.COMMENT_NODE:
                    appendCommentDetail(buf, aNode);
                    break;
                case Node.PROCESSING_INSTRUCTION_NODE:
                    appendProcessingInstructionDetail(buf, aNode);
                    break;
                case Node.DOCUMENT_TYPE_NODE:
                    appendDocumentTypeDetail(buf, aNode);
                    break;
                case Node.DOCUMENT_NODE:
                    appendDocumentDetail(buf);
                    break;
                default:
                    buf.append("!--NodeType ").append(aNode.getNodeType())
                            .append(' ').append(aNode.getNodeName())
                            .append('/').append(aNode.getNodeValue())
                            .append("--");

            }
            buf.append(XMLConstants.CLOSE_NODE);

            return buf.toString();
        }
    }

    public static <T extends ru.yandex.qatools.pagediffer.document.Document> DocumentPatch<T>
    patch(T originalDocument, T modifiedDocument) {
        List<String> originalDocumentLines = originalDocument.canonize().getLines();
        List<String> modifiedDocumentLines = modifiedDocument.canonize().getLines();
        Patch patch = DiffUtils.diff(originalDocumentLines, modifiedDocumentLines);
        return new DocumentPatch<T>(patch);
    }
}
