package ru.yandex.autotests.innerpochta.wmi.core.matchers.messages;

import java.util.List;

import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeMatcher;

import ru.yandex.autotests.innerpochta.beans.Envelope;
import ru.yandex.autotests.innerpochta.beans.hound.Messages;
import ru.yandex.autotests.innerpochta.beans.hound.Threads;
import ru.yandex.autotests.innerpochta.beans.hound.V2TabsResponse;
import ru.yandex.autotests.innerpochta.wmi.core.obj.hound.FoldersObj;
import ru.yandex.autotests.innerpochta.wmi.core.oper.hound.Folders;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;

import static java.util.function.Function.identity;
import static org.cthul.matchers.object.CIs.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.innerpochta.wmi.core.api.CommonApiSettings.shouldBe;
import static ru.yandex.autotests.innerpochta.wmi.core.api.WmiApis.apiHound;
import static ru.yandex.autotests.innerpochta.wmi.core.api.WmiApis.apiHoundV2;
import static ru.yandex.autotests.innerpochta.wmi.core.hound.HoundResponses.ok200;
import static ru.yandex.autotests.innerpochta.wmi.core.oper.hound.Folders.folders;

public abstract class IsThereMessagesMatcher extends TypeSafeMatcher<HttpClientManagerRule> {
    @Factory
    public static IsThereMessagesMatcher hasMsgIn(String subj, String fid) {
        return new IsThereMessagesWithSubjectMatcher(subj, equalTo(1), fid);
    }

    @Factory
    public static IsThereMessagesMatcher hasMsgsIn(Integer count, String fid) {
        return new IsThereMessagesInFolderMatcher(equalTo(count), fid);
    }

    @Factory
    public static IsThereMessagesMatcher hasMsgsIn(String fid) {
        return new IsThereMessagesInFolderMatcher(greaterThan(0), fid);
    }

    @Factory
    public static IsThereMessagesMatcher
    hasMsgsIn(String subj, Integer count, String fid) {
        return new IsThereMessagesWithSubjectMatcher(subj, equalTo(count), fid);
    }

    @Factory
    public static IsThereMessagesMatcher hasMsgsStrictSubjectIn(String subj, Integer count, String fid) {
        return new IsThereMessagesWithSubjectStrictMatcher(subj, equalTo(count), fid);
    }

    @Factory
    public static TypeSafeMatcher<HttpClientManagerRule> hasNewMsgsInFid(final Matcher<Integer> count, final String fid) {
        return new TypeSafeMatcher<HttpClientManagerRule>() {
            int newMsgs = 0;

            @Override
            public void describeTo(Description description) {
                description.appendText("Обнаружено новых сообщений: ").appendValue(newMsgs)
                        .appendText(" в папке ")
                        .appendValue(fid);
            }

            @Override
            protected boolean matchesSafely(HttpClientManagerRule rule) {
                newMsgs = apiHound(rule.account().userTicket()).messagesUnreadByFolder()
                        .withUid(rule.account().uid())
                        .withCount("100")
                        .withFirst("0")
                        .withFid(fid)
                        .get(shouldBe(ok200())).then()
                        .extract().body().as(Messages.class).getEnvelopes().size();

                return count.matches(newMsgs);
            }
        };
    }

    @Factory
    public static TypeSafeMatcher<HttpClientManagerRule> hasCountMsgsInFid(final Matcher<Integer> matcher, final String fid) {
        return new TypeSafeMatcher<HttpClientManagerRule>() {
            int count = 0;

            @Override
            protected boolean matchesSafely(HttpClientManagerRule rule) {
                count = Folders.folders(
                        FoldersObj.empty()
                            .setUid(rule.account().uid())
                )
                        .get()
                        .via(rule)
                        .count(fid);

                return matcher.matches(count);
            }

            @Override
            public void describeTo(Description description) {
                description.appendText("Обнаружено: ")
                        .appendValue(count)
                        .appendText(" в папке ")
                        .appendValue(fid);
            }
        };
    }

    @Factory
    public static TypeSafeMatcher<HttpClientManagerRule> hasNewMsgsInTab(final Matcher<Integer> count, final String tab) {
        return new TypeSafeMatcher<HttpClientManagerRule>() {
            int newMsgs = 0;

            @Override
            public void describeTo(Description description) {
                description.appendText("Обнаружено новых сообщений: ").appendValue(newMsgs)
                        .appendText(" в табе ")
                        .appendValue(tab);
            }

            @Override
            protected boolean matchesSafely(HttpClientManagerRule rule) {
                newMsgs = apiHoundV2(rule.account().userTicket()).tabs()
                        .withUid(rule.account().uid())
                        .get(identity()).then()
                        .extract().body().as(V2TabsResponse.class)
                        .getTabs().stream()
                        .filter(t -> t.getType().equals(tab))
                        .map(t -> t.getUnreadMessagesCount().intValue())
                        .findFirst().get();

                return count.matches(newMsgs);
            }
        };
    }

    @Factory
    public static TypeSafeMatcher<HttpClientManagerRule> hasCountMsgsInTab(final Matcher<Integer> matcher, final String tab) {
        return new TypeSafeMatcher<HttpClientManagerRule>() {
            int count = 0;

            @Override
            protected boolean matchesSafely(HttpClientManagerRule rule) {
                count = apiHoundV2(rule.account().userTicket()).tabs()
                        .withUid(rule.account().uid())
                        .get(identity()).then()
                        .extract().body().as(V2TabsResponse.class)
                        .getTabs().stream()
                        .filter(t -> t.getType().equals(tab))
                        .map(t -> t.getMessagesCount().intValue())
                        .findFirst().get();

                return matcher.matches(count);
            }

            @Override
            public void describeTo(Description description) {
                description.appendText("Обнаружено: ")
                        .appendValue(count)
                        .appendText(" в табе ")
                        .appendValue(tab);
            }
        };
    }

    @Factory
    public static TypeSafeMatcher<HttpClientManagerRule> hasMsgsWithLabel(final Matcher<Integer> count, final String lid) {
        return new TypeSafeMatcher<HttpClientManagerRule>() {
            int msgsCount = 0;

            @Override
            public void describeTo(Description description) {
                description.appendText("Обнаружено ").appendValue(msgsCount)
                        .appendText(" сообщений c меткой ")
                        .appendValue(lid);
            }

            @Override
            protected boolean matchesSafely(HttpClientManagerRule rule) {
                msgsCount = apiHound(rule.account().userTicket()).messagesByLabel()
                        .withUid(rule.account().uid())
                        .withCount("100")
                        .withFirst("0")
                        .withLid(lid)
                        .get(shouldBe(ok200())).then()
                        .extract().body().as(Messages.class).getEnvelopes().size();

                return count.matches(msgsCount);
            }
        };
    }

    @Factory
    public static TypeSafeMatcher<HttpClientManagerRule> hasThreadsInTab(final Matcher<Integer> count, final String tab) {
        return new TypeSafeMatcher<HttpClientManagerRule>() {
            int msgsCount = 0;

            @Override
            public void describeTo(Description description) {
                description.appendText("Обнаружено ").appendValue(msgsCount)
                        .appendText(" тредов в табе ")
                        .appendValue(tab);
            }

            @Override
            protected boolean matchesSafely(HttpClientManagerRule rule) {
                msgsCount = apiHoundV2(rule.account().userTicket()).threadsByTab()
                        .withUid(rule.account().uid())
                        .withTab(tab)
                        .withCount("100")
                        .withFirst("0")
                        .get(shouldBe(ok200())).then()
                        .extract().body().as(Threads.class).getThreadsByFolder().getEnvelopes().size();

                return count.matches(msgsCount);
            }
        };
    }

}

class IsThereMessagesInFolderMatcher extends IsThereMessagesMatcher {
    protected String fid;
    protected Matcher<Integer> expectedCount;

    IsThereMessagesInFolderMatcher(Matcher<Integer> count, String fid_) {
        assertThat(fid_, is(notNullValue()));
        assertThat(count, is(notNullValue()));

        fid = fid_;
        expectedCount = count;
    }

    @Override
    protected boolean matchesSafely(HttpClientManagerRule rule) {
        Integer count = folders(FoldersObj.empty()
                .setUid(rule.account().uid()))
                .get()
                .via(rule)
                .count(fid);

        return expectedCount.matches(count);
    }

    @Override
    public void describeTo(Description description) {
        description.appendText("Ожидалось сообщений - ")
                .appendDescriptionOf(expectedCount)
                .appendText(" в папке ")
                .appendText(fid);
    }
}

class IsThereMessagesWithSubjectMatcher extends IsThereMessagesInFolderMatcher {
    protected String subject;

    IsThereMessagesWithSubjectMatcher(String subj, Matcher<Integer> count, String fid) {
        super(count, fid);
        assertThat(subj, is(notNullValue()));
        subject = subj;
    }

    protected List<Envelope> getMessagesByFolder(HttpClientManagerRule rule) {
        return apiHound(rule.account().userTicket()).messagesByFolder()
                .withUid(rule.account().uid())
                .withCount("100")
                .withFirst("0")
                .withFid(fid)
                .get(shouldBe(ok200())).then()
                .extract().body().as(Messages.class).getEnvelopes();
    }

    @Override
    protected boolean matchesSafely(HttpClientManagerRule rule) {
        Integer count = Math.toIntExact(getMessagesByFolder(rule).stream()
                .filter(e -> e.getSubjectInfo().getSubject().equals(subject))
                .count());

        return expectedCount.matches(count);
    }

    @Override
    public void describeTo(Description description) {
        description.appendText("Ожидалось сообщений - ")
                .appendDescriptionOf(expectedCount)
                .appendText(" с темой ")
                .appendValue(subject.isEmpty() ? "\"\"" : subject)
                .appendText(" в папке ")
                .appendText(fid);
    }
}

class IsThereMessagesWithSubjectStrictMatcher extends IsThereMessagesWithSubjectMatcher {
    @Override
    protected boolean matchesSafely(HttpClientManagerRule rule) {
        Integer count = Math.toIntExact(getMessagesByFolder(rule).stream()
                .filter(e -> e.getSubject().equals(subject))
                .count());

        return expectedCount.matches(count);
    }

    IsThereMessagesWithSubjectStrictMatcher(String subj, Matcher<Integer> count, String fid) {
        super(subj, count, fid);
    }
}
