package ru.yandex.autotests.innerpochta.wmi.core.matchers.ssh;

import ch.ethz.ssh2.Connection;
import org.apache.log4j.Logger;
import org.cthul.matchers.chain.AndChainMatcher;
import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeMatcher;

import java.io.IOException;
import java.util.*;

import static com.google.common.base.Splitter.onPattern;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toMap;
import static org.apache.log4j.Logger.getLogger;
import static org.hamcrest.Matchers.allOf;
import static ru.yandex.autotests.innerpochta.util.ssh.SSHCommands.fgrep;

/**
 * User: angrybird
 * Date: 14.05.15
 * Time: 13:57
 */
public class TSKVLogMatcher extends TypeSafeMatcher<Connection> {
    private static Logger LOG = getLogger(TSKVLogMatcher.class);

    private String fileToGrep;
    private List<String> patterns;
    private Map<String, String> logMap = new HashMap<>();
    private List<Matcher<Map<? extends String, ? extends String>>> matchers;
    private Integer entry = 0;

    private TSKVLogMatcher(List<Matcher<Map<? extends String, ? extends String>>> matchers,
                           String fileToGrep, String pattern) {
        this.matchers = matchers;
        this.fileToGrep = fileToGrep;
        this.patterns = Arrays.asList(pattern);
    }

    private TSKVLogMatcher(List<Matcher<Map<? extends String, ? extends String>>> matchers,
                           String fileToGrep, List<String> patterns) {
        this.matchers = matchers;
        this.fileToGrep = fileToGrep;
        this.patterns = patterns;
    }

    private TSKVLogMatcher(List<Matcher<Map<? extends String, ? extends String>>> matchers,
                           String fileToGrep, String pattern, Integer entry) {
        this.matchers = matchers;
        this.fileToGrep = fileToGrep;
        this.patterns = Arrays.asList(pattern);
        this.entry = entry;
    }

    private TSKVLogMatcher(List<Matcher<Map<? extends String, ? extends String>>> matchers,
                           String fileToGrep, List<String> patterns, Integer entry) {
        this.matchers = matchers;
        this.fileToGrep = fileToGrep;
        this.patterns = patterns;
        this.entry = entry;
    }

    @Factory
    public static TSKVLogMatcher logEntryShouldMatch(List<Matcher<Map<? extends String, ? extends String>>> matchers,
                                                    String fileToGrep, String pattern) {
        return new TSKVLogMatcher(matchers, fileToGrep, pattern);
    }

    @Factory
    public static TSKVLogMatcher lastLogEntryShouldMatch(List<Matcher<Map<? extends String, ? extends String>>> matchers,
                                                     String fileToGrep, String pattern) {
        return new TSKVLogMatcher(matchers, fileToGrep, pattern, -1);
    }

    @Factory
    public static TSKVLogMatcher logEntryShouldMatch(List<Matcher<Map<? extends String, ? extends String>>> matchers,
                                                     String fileToGrep, List<String> patterns) {
        return new TSKVLogMatcher(matchers, fileToGrep, patterns);
    }

    @Factory
    public static TSKVLogMatcher logEntryShouldMatch(List<Matcher<Map<? extends String, ? extends String>>> matchers,
                                                     String fileToGrep, String pattern, Integer entry) {
        return new TSKVLogMatcher(matchers, fileToGrep, pattern, entry);
    }

    @Factory
    public static TSKVLogMatcher logEntryShouldMatch(List<Matcher<Map<? extends String, ? extends String>>> matchers,
                                                     String fileToGrep, List<String> patterns, Integer entry) {
        return new TSKVLogMatcher(matchers, fileToGrep, patterns, entry);
    }

    private static Map<String, String> parse(String line) {
        return onPattern("\\s+").omitEmptyStrings().splitToList(line).stream()
                .map(o -> o.split("=", 2))
                .collect(toMap(
                        arr -> arr[0],
                        arr -> arr.length > 1 ? arr[1] : ""
                ));
    }

    @Override
    protected boolean matchesSafely(Connection conn) {
        try {
            final String firstPattern = patterns.isEmpty() ? "" : patterns.get(0);
            ArrayList<String> logs = fgrep(conn, LOG, firstPattern, fileToGrep);
            patterns.stream().skip(1).forEach(pattern -> {
                logs.removeIf(log -> !log.contains(pattern));
            });
            if (entry < 0) {
                entry += logs.size();
            }
            logMap = logs.isEmpty() ? emptyMap() : parse(logs.get(entry));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return AndChainMatcher.and(matchers).matches(logMap);
    }

    @Override
    public void describeTo(Description description) {
        description.appendDescriptionOf(AndChainMatcher.and(matchers));
        description.appendText("\nПолученные в лог записи: \n").appendText(logMap.toString());
    }

    @Override
    protected void describeMismatchSafely(Connection actual,
                                          Description mismatchDescription) {
        allOf(matchers.toArray(new Matcher[]{})).describeMismatch(logMap, mismatchDescription);
    }

}
