package ru.yandex.autotests.innerpochta.wmi.core.obj;


import com.google.common.base.Charsets;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.entity.BasicHttpEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.message.BasicNameValuePair;
import org.apache.tools.ant.filters.StringInputStream;

import java.io.File;
import java.nio.charset.Charset;
import java.util.*;

import static org.apache.commons.lang3.Validate.isTrue;
import static org.hamcrest.Matchers.*;

/**
 * Created by IntelliJ IDEA.
 * User: lanwen
 * Date: 21.03.12
 * Time: 16:53
 * Базовый класс для всех наборов свойств
 */
public abstract class Obj {

    protected HashMap<String, List<String>> queryParams = new HashMap<>();
    protected HashMap<String, List<String>> params = new HashMap<>();
    protected HashMap<String, List<FileBody>> files = new HashMap<>();
    protected String content;

    public Obj add(boolean isQueryParam, String name, String... values) {
        isTrue(not(hasItemInArray(equalTo(null))).matches(values), "Значение параметра %s не может быть null", name);
        if (isQueryParam) {
            addToMap(queryParams, name, values);
        } else {
            addToMap(params, name, values);
        }
        return this;
    }

    private void addToMap(Map<String, List<String>> map, String name, String... values) {
        if (!map.containsKey(name)) {
            map.put(name, Arrays.asList(values));
        } else {
            List<String> currVal = new ArrayList<String>();
            currVal.addAll(map.get(name));
            currVal.addAll(Arrays.asList(values));
            map.put(name, currVal);
        }
    }

    public Obj add(String name, String... values) {
        return add(false, name, values);
    }

    public Obj set(String name, String... values) {
        remove(name);
        return add(false, name, values);
    }

    public Obj set(boolean onlyQuery, String name, String... values) {
        remove(name);
        return add(onlyQuery, name, values);
    }

    public Obj setContent(String content) {
        this.content = content;
        return this;
    }


    public Obj addFile(String name, String mimetype, File... attas) {

        List<FileBody> fileBodies;

        if (files.containsKey(name)) {
            fileBodies = files.get(name);
        } else {
            fileBodies = new ArrayList<FileBody>();
        }

        for (File file : attas) {
            fileBodies.add(new FileBody(file, mimetype, "UTF-8"));
        }
        files.put(name, fileBodies);
        return this;
    }

    public Obj addFile(String name, File... file) {
        return addFile(name, "binary/octet-stream", file);
    }


    public String onlyQueryParams(boolean addQuestion) {
        return queryParams.isEmpty() ? "" : getMapAsString(queryParams, addQuestion);
    }

    public String asGet(boolean addQuestion) {
        Map<String, List<String>> map = new HashMap<>();

        map.putAll(queryParams);
        map.putAll(params);

        return map.isEmpty() ? "" : getMapAsString(map, addQuestion);
    }

    /**
     * Возвращает http параметры в виде строки, без & в начале:
     *
     * @return
     * @example
     */
    public String asGet() {
        return asGet(false).replaceFirst("&", "");
    }

    public HttpEntity asPost() {
        if (hasBody()) {
            return getBasicHttpEntity(content);
        }

        MultipartEntity entity = new MultipartEntity(HttpMultipartMode.BROWSER_COMPATIBLE, null, Charsets.UTF_8);

        for (String key : params.keySet()) {
            for (String value : params.get(key)) {
                entity.addPart(key, StringBody.create(value, "text/plain", Charset.forName("UTF-8")));
            }
        }

        for (String key : files.keySet()) {
            for (FileBody body : files.get(key)) {
                entity.addPart(key, body);
            }
        }

        return entity;
    }

    private BasicHttpEntity getBasicHttpEntity(String content) {
        BasicHttpEntity entity = new BasicHttpEntity();
        entity.setContent(new StringInputStream(content, "UTF-8"));
        entity.setContentLength(content.getBytes(Charsets.UTF_8).length);
        return entity;
    }

    public boolean hasBody() {
        return content != null;
    }

    public String getBody() {
        return content;
    }

    public String getParam(String key) {
        Map<String, List<String>> map = new HashMap<>();

        map.putAll(queryParams);
        map.putAll(params);

        if (null == map.get(key) || map.get(key).size() == 0) {
            return null;
        } else {
            return map.get(key).get(0);
        }
    }


    public List<String> getParamAsList(String key) {
        Map<String, List<String>> map = new HashMap<>();

        map.putAll(queryParams);
        map.putAll(params);

        return map.get(key);
    }


    public Obj clear() {
        params.clear();
        queryParams.clear();
        content = null;
        return this;
    }

    public Obj remove(String key) {
        params.remove(key);
        queryParams.remove(key);
        files.remove(key);
        return this;
    }


    private String getMapAsString(Map<String, List<String>> map, boolean addQuestion) {
        StringBuilder sb = new StringBuilder();
        if (addQuestion) {
            sb.append("?");
        } else {
            sb.append("&");
        }

        for (String key : map.keySet()) {
            for (String value : map.get(key)) {
                if (sb.length() > 1) { // && !sb.toString().equals("&") && !sb.toString().equals("?")
                    sb.append("&");
                }
                if (!value.isEmpty()) {
                    sb.append(key).append("=").append(value);
                } else {
                    sb.append(key);
                }

            }
        }
        return sb.toString();
    }


    public ArrayList<NameValuePair> nameValuePairs() {
        Map<String, List<String>> map = new HashMap<>();

        map.putAll(queryParams);
        map.putAll(params);

        ArrayList<NameValuePair> pairsArrayList = new ArrayList<NameValuePair>(map.size());

        for (String key : map.keySet()) {
            for (String value : map.get(key)) {
                pairsArrayList.add(new BasicNameValuePair(key, value));
            }
        }

        return pairsArrayList;
    }


    /**
     * В каком виде отображать
     *
     * @param xmlVersion - вид отображения
     *                   Бывает: daria2, default ...
     * @return MailboxListNearestInThreadObj
     */
    public Obj setXmlVersion(String xmlVersion) {
        add("xml_version", xmlVersion);
        return this;
    }

    //------------------------------------------------------
    public static final String XMLVERSION_DARIA2 = "daria2";  // Используется neo2
    public static final String XMLVERSION_DEFAULT = "default";
    //------------------------------------------------------


}
