package ru.yandex.autotests.innerpochta.wmi.core.oper;

import org.apache.commons.lang3.Validate;
import org.json.JSONException;
import org.json.JSONObject;
import org.w3c.dom.NodeList;
import ru.yandex.autotests.innerpochta.beans.folderlist.Symbol;
import ru.yandex.autotests.innerpochta.objstruct.oper.OperClass;
import ru.yandex.autotests.innerpochta.wmicommon.WmiConsts;

import java.io.IOException;
import java.util.List;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.Matchers.isEmptyOrNullString;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.innerpochta.beans.folderlist.Symbol.OUTBOX;

/**
 * Created by IntelliJ IDEA.
 * User: lanwen
 * Date: 22.03.12
 * Time: 17:02
 * <p>
 * Операция списка папок
 * Наследники:
 * - AllFolderList - т.к. очень похожа
 * * Можно вызывать без назначенного объекта
 */
@OperClass(
        apicommand = "folder_list",
        jsxcommand = "folder_list",
        description = "Вывод списка папок"
)
public class FolderList extends Oper<FolderList> {


    /**
     * Проверяет существует ли папка с заданным именем в списке
     *
     * @param nameToFind - имя папки для поиска
     * @return - true если папка с заданным именем существует
     */
    public Boolean isThereFolder(String nameToFind) {
        NodeList nodes = documentConverter.byXpath("//folder[name='" + nameToFind + "']").asNodeList();
        return nodes.item(0) != null;
    }

    public void emptyFolder(String folderName) throws IOException {
        assertThat("Папка не пустая ", this.getFolderCount(folderName),
                equalTo(0));
    }


    /**
     * Получает fid папки с заданным именем
     *
     * @param name - имя папки
     * @return - String fid
     */
    public String getFolderId(String name) {
        String fid = documentConverter.byXpath("//folder[name='" + name + "']/fid/text()").asString();
        assertThat("В folder_list нет папки с именем " + name, fid, notNullValue());
        return fid;
    }

    /**
     * Получает fid папки с заданным символом
     */
    public String fidBySymbol(Symbol symbol) {
        final String fid = documentConverter.byXpath(String.format("//folder[symbol='%s']/fid/text()", symbol.toString().toLowerCase())).asString();
        assertThat("Не нашли fid для папки с символом " + symbol, fid, not(isEmptyOrNullString()));
        return fid;
    }

    public String nameBySymbolOrNull(Symbol symbol) {
        return documentConverter.byXpath(String.format("//folder[symbol='%s']/name/text()", symbol.toString().toLowerCase())).asString();
    }

    public String nameBySymbol(Symbol symbol) {
        final String name = nameBySymbolOrNull(symbol);
        assertThat("Не нашли name для папки с символом " + symbol, name, not(isEmptyOrNullString()));
        return name;
    }

    /**
     * Получаем количество писем в папке с заданным именем
     *
     * @param name - имя папки
     * @return - Integer count
     */
    public Integer getFolderCount(String name) {
        return documentConverter.byXpath("//folder[name='" + name + "']/count/text()").asInteger();
    }

    /**
     * Получаем количество новых писем в папке с заданным именем
     *
     * @param name - имя папки
     * @return - Integer count
     */
    public Integer getFolderNew(String name) {
        return documentConverter.byXpath("//folder[name='" + name + "']/new/text()").asInteger();
    }

    public String getFolderPop3(String fid) {
        return documentConverter.byXpath("//folder[fid='" + fid + "']/pop3on/text()").asString();
    }

    public FolderList pop3Yes(String fid) {
        assertThat(String.format("Неверное значение у аттрибута <pop3on> у папки <%s>", fid), getFolderPop3(fid), equalTo("yes"));
        return this;
    }

    public FolderList pop3No(String fid) {
        assertThat(String.format("Неверное значение у аттрибута <pop3on> у папки <%s>", fid), getFolderPop3(fid), equalTo("no"));
        return this;
    }

    public String getFolderName(String fid) {
        return documentConverter.byXpath("//folder[fid='" + fid + "']/name/text()").asString();
    }

    /**
     * fid дефолтной папки
     *
     * @return фид дефолтной папки
     */
    public String defaultFID() {
        return fidBySymbol(Symbol.INBOX);
    }

    /**
     * fid папки отправленные
     *
     * @return фид дефолтной папки
     */
    public String sentFID() {
        return fidBySymbol(Symbol.SENT);
    }

    /**
     * fid папки черновики
     *
     * @return фид дефолтной папки
     */
    public String draftFID() {
        return fidBySymbol(Symbol.DRAFT);
    }

    public String templateFID() {
        return fidBySymbol(Symbol.TEMPLATE);
    }

    public String outgoingFID() {
        return fidBySymbol(Symbol.OUTBOX);
    }

    /**
     * fid папки спам
     *
     * @return фид дефолтной папки
     */
    public String spamFID() {
        return fidBySymbol(Symbol.SPAM);
    }

    /**
     * fid папки удаленные
     *
     * @return фид дефолтной папки
     */
    public String deletedFID() {
        return fidBySymbol(Symbol.TRASH);
    }


    /**
     * имя дефолтной папки
     *
     * @return фид дефолтной папки
     */
    public String defaultName() {
        return nameBySymbol(Symbol.INBOX);
    }

    /**
     * имя папки отправленные
     *
     * @return фид дефолтной папки
     */
    public String sentName() {
        return nameBySymbol(Symbol.SENT);
    }

    /**
     * имя папки черновики
     *
     * @return фид дефолтной папки
     */
    public String draftName() {
        return nameBySymbol(Symbol.DRAFT);
    }

    public String templateName() {
        return nameBySymbol(Symbol.TEMPLATE);
    }

    public String outgoingName() {
        return nameBySymbol(Symbol.OUTBOX);
    }

    /**
     * имя папки спам
     *
     * @return фид дефолтной папки
     */
    public String spamName() {
        return nameBySymbol(Symbol.SPAM);
    }

    /**
     * имя папки удаленные
     *
     * @return фид дефолтной папки
     */
    public String deletedName() {
        return nameBySymbol(Symbol.TRASH);
    }


    /**
     * Получает значение внутри тега folder_options, десериализованное в JSON объект
     *
     * @param fid фид папки
     * @return json obj
     */
    public JSONObject getFolderOptions(String fid) {
        String foptions = null;
        try {
            foptions = documentConverter.byXpath("//folder[fid='" + fid + "']/folder_options/text()").asString();

            //Логирование
            logger.info(String.format("%s(%s) folder_options: %s", getFolderName(fid), fid, foptions));

            return new JSONObject(foptions);
        } catch (JSONException e) {
            logger.fatal("Недопустимое значение внутри <folder_options>: " + foptions);
            e.printStackTrace();
        }
        return null;
    }


    /**
     * Получает значение position
     *
     * @param fid фид папки
     * @return json obj
     */
    public Integer getPosition(String fid) throws JSONException {
        JSONObject jsonObject = getFolderOptions(fid);
        return Integer.parseInt(jsonObject.get("position").toString());
    }


    /**
     * Возвращает значение тега, символизирующего изменялась ли папка с последнего просмотра
     *
     * @param fid - фид папки
     * @return строковое значение тега
     */
    public String getFolderRecentValue(String fid) {

        String recent = documentConverter.byXpath("//folder[fid = " + fid + "]/recent/text()").asString();

        // Логирование
        logger.info("Recent value for fid: " + fid + " - " + recent);

        return recent;
    }

    /**
     * Проверяет наличие подпапки
     *
     * @param name       - имя искомой подпапки
     * @param parentName - имя родительской папки
     * @return - true, если есть такая папка
     */
    //Регулярное выражение: byXpath("//folder[@name='*|{0,1}" + parentName + "|" + name + "$']");
    public Boolean isThereSubFolder(String name, String parentName) {
        String pattern = parentName + "|" + name;
        //временный вариант возможно проблемы если патерн будет располагаться посередине
        NodeList nodes = documentConverter.byXpath("//folder/name[contains(.,\"" + pattern + "\")]").asNodeList();
        //TODO: подключить новый xpath, старый не понимает matches и регуляные выражения
        return nodes.item(0) != null;
    }

    /**
     * @param name       - имя искомой подпапки
     * @param parentName - имя родительской папки
     * @return fid подпапки
     */
    public String getFidSubfolder(String name, String parentName) {
        String pattern = parentName + "|" + name;
        return documentConverter.byXpath("//folder[contains(name,\"" + pattern + "\")]/fid/text()").asString();
    }

    /**
     * Получение списка всех папок, исключая системные
     *
     * @return List<String> - список фидов папок, исключая некоторые системные
     */
    public List<String> getAllFolderIdsWithoutSystem() {
        return documentConverter.byXpath("//folder" +
                "[not(name/text()='" + defaultName() + "') " +
                "and not(name/text()='" + sentName() + "') " +
                "and not(name/text()='" + deletedName() + "') " +
                "and not(name/text()='" + nameBySymbolOrNull(OUTBOX) + "') " +
                "and not(name/text()='" + spamName() + "') " +
                "and not(name/text()='Рассылки') " +                 // странный баг с отображением названия спам
                // как "Рассылки"
                "and not(name/text()='" + draftName() + "')]" +
                "/fid/text()").asList();
    }

    /**
     * Получение списка всех папок, исключая системные
     *
     * @return List<String> - список имен папок, исключая некоторые системные
     */
    public List<String> getAllFolderNamesWithoutSystem() {
        return documentConverter.byXpath("//folder" +
                "[not(name/text()='" + defaultName() + "') " +
                "and not(name/text()='" + sentName() + "') " +
                "and not(name/text()='" + deletedName() + "') " +
                "and not(name/text()='" + WmiConsts.FOLDER_DELAYED + "') " +
                "and not(name/text()='" + spamName() + "') " +
                "and not(name/text()='Рассылки') " +                 // странный баг с отображением названия спам
                // как "Рассылки"
                "and not(name/text()='" + draftName() + "')]" +
                "/name/text()").asList();
    }


    /**
     * Получение списка всех папок
     *
     * @return List<String> - список фидов
     */
    public List<String> getAllFolderIds() {
        return documentConverter.byXpath("//fid/text()").asList();
    }

    /**
     * Получение списка имен всех папок
     *
     * @return List<String> - список имен всех папок
     */
    public List<String> getAllFolderNames() {
        return documentConverter.byXpath("//name/text()").asList();
    }


    /**
     * Получение списка значений тега symbol всех папок
     *
     * @return List<String> - список папок
     */
    public List<String> getAllFolderSymbolTag() {
        return documentConverter.byXpath("//folder/symbol/text()").asList();
    }


    public String getSpecial(String fid) {
        return documentConverter.byXpath("//folder[fid='" + fid + "']/special").asString();
    }


    public String getSymbol(String fid) {
        return documentConverter.byXpath("//folder[fid='" + fid + "']/symbol").asString();
    }
}
