package ru.yandex.autotests.innerpochta.wmi.core.oper;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import ru.yandex.autotests.innerpochta.objstruct.oper.OperClass;
import ru.yandex.autotests.innerpochta.wmi.core.obj.Obj;
import ru.yandex.autotests.innerpochta.wmi.core.response.MailboxListResponseItem;

import javax.xml.xpath.XPathExpressionException;
import java.util.List;

import static java.lang.String.format;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.jsx;

/**
 * Created by IntelliJ IDEA.
 * User: lanwen
 * Date: 22.03.12
 * Time: 17:02
 * <p/>
 * Операция списка папок
 * Можно использовать без параметров
 */
@OperClass(
        apicommand = "mailbox_list",
        jsxcommand = "mailbox_list",
        description = "Вывод содержимого папки"
)
public class MailBoxList extends Oper<MailBoxList> {
    
    public static MailBoxList mailboxListJsx(Obj params) {
        return mailboxListJsx().params(params);
    }

    public static MailBoxList mailboxListJsx() {
        return jsx(MailBoxList.class);
    }


    /**
     * Возвращает mid первого найденного письма с заданной темой
     *
     * @param subject - тема письма
     * @return String mid - мид письма в виде строки
     */
    public String getMidOfMessage(String subject) {
        String mid = documentConverter.byXpath("//message[subject/text='" + subject + "']/@mid").asString();

        // Логирование значения
        if (mid != null) {
            logger.info("MID первого найденного письма c темой: " + subject + " - " + mid);
        } else {
            logger.error("Письмо '" + subject + "' не найдено!");
        }

        return mid;
    }


    /**
     * Возвращает тему первого найденного письма с заданным мид
     *
     * @param mid - мид
     * @return String mid - тема письма в виде строки
     */
    public String getSubjectOfMessage(String mid) {
        return documentConverter.byXpath("//message[@mid/text='" + mid + "']/subject").asString();
    }

    /**
     * Возвращает stid первого письма с заданной темой
     *
     * @param subject - тема
     * @return String stid - stid письма в виде строки
     */
    public String getStidOfMessage(String subject) {
        String stid = documentConverter.byXpath("//message[subject/text='" + subject + "']/@st_id").asString();

        // Логирование значения
        if (stid != null) {
            logger.info("stid первого найденного письма c темой: " + subject + " - " + stid);
        } else {
            logger.error("Письмо '" + subject + "' не найдено!");
        }

        return stid;
    }

    /**
     * Проверяет наличие письма с заданной темой
     *
     * @param subject - Тема письма
     * @return - true если письмо найдено
     */
    public boolean isThereMessage(String subject) {
        NodeList nodes = documentConverter.byXpath("//message[subject/text='" + subject + "']").asNodeList();
        return nodes.item(0) != null;
    }

    /**
     * Возвращает количество писем с данной темой в выдаче mailbox_list
     *
     * @param subject - тема писем
     * @return Целое значение количества
     */
    public Integer countMessagesInFolderWithSubj(String subject) {
        NodeList nodes = documentConverter.byXpath("//message[subject/text='" + subject + "']").asNodeList();
        return nodes.getLength();
    }

    /**
     * Возвращает дату отправки письма recv_date
     *
     * @param mid - мид письма
     * @return Дата отправки
     */
    public String getRecvDate(String mid) {

        String recvDate = documentConverter.byXpath("//message[@mid='" + mid + "']/@recv_date").asString();

        //Log
        logger.info("Дата отправки: " + recvDate);

        return recvDate;
    }

    /**
     * Возвращает дату отправки письма recv_date, поиск по subject
     *
     * @param subject - тема письма
     * @return Дата отправки
     */
    public String getRecvDateBySubject(String subject) {
        String recvDate = documentConverter.byXpath("//message[subject/text='" + subject + "']/@recv_date").asString();
        logger.info("Дата отправки: " + recvDate);
        return recvDate;
    }

    public long getTimestamp(String mid) {
        String timestamp = documentConverter.byXpath("//message[@id='" + mid + "']/date/timestamp").asString();
        logger.info("timestamp: " + timestamp);
        return Long.parseLong(timestamp) / 1000;
    }

    public long getUtcTimestamp(String mid) {

        String utcTimestamp = documentConverter.byXpath("//message[@id='" + mid + "']/date/utc_timestamp").asString();
        logger.info("utc_timestamp: " + utcTimestamp);
        return Long.parseLong(utcTimestamp);
    }

    /**
     * Возвращает количество любых отображенных писем в папке
     * Отображенных = если меньше настройки отображения, то реальное кол-во,
     * если больше настройки, то настройку
     * Для более точного числа во всей папке, нужно использовать getMessageCounter()
     *
     * @return Целочисленное количество отображенных писем в папке
     */
    public Integer countMessagesInFolder() {
        NodeList nodes = documentConverter.byXpath("//message").asNodeList();
        return nodes.getLength();
    }

    /**
     * Возвращает значение атрибута msg_count
     *
     * @return Целочесленное значение атрибута
     */
    public Integer getMessageCounter() {
        NodeList nodes = documentConverter.byXpath("//mailbox_list/details/@msg_count").asNodeList();
        return Integer.parseInt(nodes.item(0).getNodeValue());
    }

    /**
     * Возвращает значение тега from - то, которое будет отображаться в верстке
     *
     * @param mid - мид письма
     * @return Имя отправителя
     */
    public String getFromNameByMid(String mid) {
        String fromName = documentConverter.byXpath("//message[@mid='" + mid + "']/from/item/name/text()").asString();

        // Логирование значения
        logger.info("Письмо от: " + fromName);

        return fromName;
    }

    /**
     * Получаем значение тега To
     * Это значение, отображаемое в верстке, в поле "Кому"
     *
     * @param mid - mid письма, для которого ищем
     * @return значение кому
     */
    public String getToNameByMid(String mid) {

        String toName = documentConverter.byXpath("//message[@mid='" + mid + "']/to/item/name/text()").asString();

        // Логирование значения
        logger.info("Кому: " + toName);

        return toName;
    }


    public String getFirstlineBySubject(String subject) {
        String firstline = documentConverter
                .byXpath("//message[subject/text='" + subject + "']/firstline/text()").asString();

        // Логирование значения
        logger.info("Первая строка: " + firstline);

        return firstline;
    }


    /**
     * Получает миды писем с заданной темой
     *
     * @param subject тема письма
     * @return Список строк - мидов писем
     */
    public List<String> getMidsOfMessagesWithSubject(String subject) {
        return documentConverter.byXpath("//message[subject/text='" + subject + "']/@mid").asList();
    }

    /**
     * Получает миды всех писем из выдачи
     *
     * @return List<String> midы писем
     */
    public List<String> getMidsOfMessagesInFolder() {
        return documentConverter.byXpath("//message/@mid").asList();
    }

    /**
     * @return - mid первого письма в выдаче
     */
    public String getMidOfFirstMessage() {
        List<String> mids = getMidsOfMessagesInFolder();
        assertThat("Для получения первого мид из списка, список должен быть не пуст", mids, hasSize(greaterThan(0)));
        String mid = mids.get(0);
        logger.info("MID первого письма: " + mid);

        return mid;
    }

    /**
     * Проверка, что письмо с указанным мидом помечено ВСЕМИ метками из списка
     *
     * @param mid    - письмо, для которого идет проверка
     * @param labels - список лидов
     * @return возвращает true только в том случае, если письмо обладает всеми метками списка
     */
    public Boolean isMessageLabeled(String mid, List<String> labels) {
        Boolean hasMid = documentConverter.byXpath(format("count(//message[@mid='%s']) > 0",
                mid)).asBoolean();
        if (!hasMid) {
            throw new IllegalArgumentException(String.format("Письмо с <MID:%s> в выдаче не обнаружено", mid));
        }

        Boolean flag = true;
        for (String currentLabel : labels) {
            boolean labelled = documentConverter.byXpath(format("count(//message[@mid='%s' and lids/item='%s']) > 0",
                    mid, currentLabel)).asBoolean();
            if (!labelled) {
                logger.info("Метка - " + currentLabel + " не найдена");
                flag = false;
            } else {
                logger.info("Есть метка - " + currentLabel);
            }
        }
        return flag;
    }

    public List<String> getLidsOfMsgWithMid(String mid) {
        return documentConverter.byXpath("//message[@mid='" + mid + "']/lids/item/text()").asList();
    }

    /**
     * Проходит все миды и для каждого смотрит, помечено ли оно всеми указанными метками
     * Если находится хоть одно письмо непомеченное - возвращаем фолс
     *
     * @param mids - Список мидов
     * @param lids - Список лидов
     * @return true если все письма помечены всеми метками
     */
    public Boolean isSomeMessagesLabeled(List<String> mids, List<String> lids) {
        for (String mid : mids) {
            if (!isMessageLabeled(mid, lids)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Просматривает вывод на предмет наличия хоть одного письма с указанной меткой
     *
     * @param labelId - лид метки для поиска
     * @return true - если нашли хоть одно письмо с данной меткой
     */
    public Boolean isThereAnyMessageWithSuchLabel(String labelId) {
        return documentConverter.byXpath("count(//lids[item='" + labelId + "']) > 0").asBoolean();
    }


    public Integer getAttCount(String mid) throws XPathExpressionException {
        return documentConverter.byXpath("//message[@mid='" + mid + "']/@att_count").asInteger();
    }


    public String getCurrentFid() {
        return documentConverter.byXpath("//details/@fid").asString();
    }

    public MailboxListResponseItem getMessageBySubject(String subject) {
        NodeList nodes = documentConverter.byXpath("//message[subject/text='" + subject + "']").asNodeList();
        if (nodes.getLength() == 0) {
            logger.info("Писем с темой '" + subject + "' не найдено");
           return null;
        } else {
            return new MailboxListResponseItem((Element)nodes.item(0));
        }
    }
}
