package ru.yandex.autotests.innerpochta.wmi.core.oper;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import org.hamcrest.Matcher;
import org.w3c.dom.NodeList;
import ru.yandex.autotests.innerpochta.objstruct.oper.OperClass;
import ru.yandex.autotests.innerpochta.wmicommon.Util;
import ru.yandex.autotests.plugins.testpers.html.common.Code;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.lang.StringUtils.abbreviate;
import static org.apache.commons.lang.StringUtils.defaultIfEmpty;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.api;
import static ru.yandex.autotests.innerpochta.wmi.core.obj.MessageObj.getMsg;
import static ru.yandex.autotests.innerpochta.wmi.core.obj.Obj.XMLVERSION_DARIA2;
import static ru.yandex.autotests.innerpochta.wmicommon.Util.convertNodeToString;

/**
 * Created by IntelliJ IDEA.
 * User: lanwen
 * Date: 22.03.12
 * Time: 17:02
 */
@OperClass(
        apicommand = "message",
        description = "Вывод конкретного письма"
)
public class Message extends Oper<Message> {


    public String bccEmail() {
        return documentConverter.byXpath("//bcc/email").asString();
    }

    public String ccEmail() {
        return documentConverter.byXpath("//cc/email").asString();
    }

    public String toEmail() {
        return documentConverter.byXpath("//to/email").asString();
    }

    public String toName() {
        return documentConverter.byXpath("//to/name").asString();
    }

    public String getToName() {
        NodeList nodes = documentConverter.byXpath("//to/item/name/text()").asNodeList();
        return nodes.item(0).getNodeValue();
    }

    public String getFromName() {
        NodeList nodes = documentConverter.byXpath("//from/item/name/text()").asNodeList();
        return nodes.item(0).getNodeValue();
    }

    public List<String> toEmailList() {
        return documentConverter.byXpath("//to/item/email/text()").asList();
    }

    public List<String> toNameList() {
        return documentConverter.byXpath("//to/item/name/text()").asList();
    }

    public String fromEmail() {
        return documentConverter.byXpath("//from/email").asString();
    }

    public String types() {
        return Joiner.on(",").join(Splitter.on("\n").split(documentConverter.byXpath("//types").asString()));
    }

    public void types(Matcher<String> matcher) {
        assertThat("Ожидалось другое значение поля <types>", types(),  matcher);
    }

    public String timestampSec() {
        return documentConverter.byXpath("//date/timestamp").asSeconds();

    }

    public String infoTimestampSec() {
        return documentConverter.byXpath("//info/date/timestamp").asSeconds();

    }

    public String firstSubTypeOf(String type) {
        return documentConverter.byXpath("//part[type='" + type + "']/subtype/text()").asString();
    }

    /**
     * Возвращает значение тега message-id
     *
     * @return Строка вида <391333101497@wmi-qa.yandex.ru> (с уголками)
     */
    public String getMessageIdTag() {
        String messageIdTag = documentConverter.byXpath("//message-id/text()").asString();

        // Логирование значения
        logger.info("Message-Id: " + messageIdTag);

        return messageIdTag;
    }

    /**
     * Возвращает значение тега st_id
     */
    public String getStId() {
        String messageIdTag = documentConverter.byXpath("//st_id/text()").asString();

        // Логирование значения
        logger.info("st_id: " + messageIdTag);

        return messageIdTag;
    }

    /**
     * Возвращает значение тега subject
     *
     * @return Тема сообщения
     */
    public String getSubject() {
        String subj = documentConverter.byXpath("//hdr_subject/text/text()").asString();

        // Логирование значения
        logger.info("Тема сообщения: " + subj);

        return subj;
    }

    /**
     * Возвращает содержимое тега facts конвертируя его в json-объект
     *
     * @return JSONObject - вида
     * {"calendar" : [{"CalendarEvent":[{"Cancel":"0"},{"Event":"III Выставка Конференция 'Кадровая служба
     * информационное обеспечение'"},{"Place2":"Московский отель 'Космос'"},{"Reschedule":"0"},
     * {"Time":"2012-04-01 00:00:00 - 2012-04-30 00:00:00"},{"Until":"0"}]}]}
     */
    public String getFormattedFacts() {
        return Code.format(documentConverter.byXpath("//facts/text()").asString());
    }


    /**
     * Аналогично getFormattedFacts, только без форматирования
     */
    public String getRawFacts() {
        return documentConverter.byXpath("//facts/text()").asString();
    }

    /**
     * Возвращает значение тега content как узла
     *
     * @return что было получено
     */
    public String getContentTag() {
        NodeList contentTag = documentConverter.byXpath("//content").asNodeList();
        return convertNodeToString(contentTag.item(0));
    }

    /**
     * Возвращает значение url инлайновой картинки
     *
     * @return что было получено
     */
    public String getImg() {
        return documentConverter.byXpath("//img/@src").asString();
    }


    /**
     * Возвращает значение тега content как узла
     * ВНИМАНИЕ! Применяется совместно с showContentMeta флагом
     * Получает внутренности, если, например, был отправлен html вида:
     * <\html\><\body\>http://te.st<\/body\><\/html\>
     *
     * @return что было получено
     */
    public String getContentTagAsSimpleHtmlResult() {
        NodeList nodes = documentConverter.byXpath("//span/text()").asNodeList();
        String result = (null != nodes.item(0))
                ? nodes.item(0).getNodeValue()
                : null;
        // Логирование значения
        //logger.info("Content: " + contentTag);

        return result;
    }

    public String getMessageFromHeader() {
        NodeList nodes = documentConverter.byXpath("//hdr_from/item/email/text()").asNodeList();
        assertNotNull("hdr_from не найден", nodes.item(0));
        return nodes.item(0).getNodeValue();
    }

    /**
     * Возвращает значение тега content
     * Возможно использование только без флага showContentMeta при получении письма
     *
     * @return что было получено
     */
    public String getTextValueOfContentTag() {
        try {
            return Util.getSingleNodeValue(toDocument(), "content");
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public String getContentTagText() {
        return documentConverter.byXpath("//content").asString();
    }


    public String getHidOfAttach(String attachName) {
        return documentConverter.log(logger, "Hid of " + attachName + " is")
                .byXpath("//part[name='" + attachName + "']/hid/text()").asString();
    }

    /**
     * Получаем идентификатор для встроенных писем (hid + .1)
     *
     * @param attachName - имя аттача, у которого получаем
     * @return
     */
    public String getBidOfAttach(String attachName) {
        return documentConverter.log(logger, "Bid of " + attachName + " is")
                .byXpath("//part[name='" + attachName + "']/@bid").asString();
    }

    public Message attachWithNameHasUrl(String attachName, String url) {
        return getFromAttachWithName(attachName, "url", url);
    }

    public Message attachWithNameHasPreview(String attachName, String preview) {
        return getFromAttachWithName(attachName, "preview", preview);
    }

    public Message attachWithNameHasSize(String attachName, String size) {
        return getFromAttachWithName(attachName, "size", size);
    }

    public Message attachWithNameHasClass(String attachName, String classAttach) {
        return getFromAttachWithName(attachName, "class", classAttach);
    }

    private Message getFromAttachWithName(String attachName, String what, String expected) {
        String result = documentConverter.log(logger, String.format("%s of %s is", what, attachName))
                .byXpath(String.format("//part[name='" + attachName + "']/%s", what)).asString();
        assertThat(String.format("Неправильное значение <%s> у аттача %s", what, attachName),
                result, equalTo(expected));
        return this;
    }

    public List<String> getAllHids() {
        return documentConverter.byXpath("//hid/text()").asList();
    }

    public String getNameOfAttachByHid(String hid) {
        String name = documentConverter.byXpath("//part[hid='" + hid + "']/name/text()").asString();

        // Логирование
        logger.info("Name of att with hid:" + hid + " is - " + name);

        return name;
    }

    /**
     * Возвращает значение тега firstline
     *
     * @return String
     */
    public String getFirstlineText() {
        return documentConverter.byXpath("//firstline/text()").asString();
    }

    /**
     * Возвращает значение тега lang - определенный язык письма в виде кода по ISO 639
     *
     * @return String
     */
    public String getLangCode() {
        return documentConverter.log(logger, "Язык с кодом").byXpath("//lang/text()").asString();
    }


    public String getLangAndText(Integer howMuchSymbols) {
        String lang = documentConverter.byXpath("//lang/text()").asString();
        String contentTag = documentConverter.byXpath("//content/text()").asString();

        return String.format("Lang: %s%n%s", lang, abbreviate(contentTag, howMuchSymbols));
    }


    /**
     * Получает hid инлайнового аттача (вроде смайлов)
     *
     * @param id - ид инлайнового аттача
     * @return строковое значение hid
     */
    public String getHidOfInlineAttach(String id) {
        String prop = getPropertyOfInlineAttach(id, "hid");

        //Логирование
        logger.info("HID аттача: " + prop);

        return prop;
    }

    /**
     * Получает имя файла инлайнового аттача (вроде смайлов)
     *
     * @param id - ид инлайнового аттача
     * @return строковое значение hid
     */
    public String getNameOfInlineAttach(String id) {
        String prop = getPropertyOfInlineAttach(id, "name");

        //Логирование
        logger.info("Имя аттача: " + prop);

        return prop;
    }

    /**
     * Получает значение необходимого свойства для inline-аттачей
     *
     * @param id       - ид аттача
     * @param property - имя свойства
     * @return - строковое значение
     */
    private String getPropertyOfInlineAttach(String id, String property) {
        String propValue = documentConverter.byXpath("//part[disposition_value='inline' and content_id='"
                + id + "']/"
                + property + "/text()").asString();

        // Логирование
        logger.info(property + " = " + propValue + " for " + id);

        return propValue;
    }

    /**
     * Возвращает список из пар отображаемое значение-ссылка
     * Выдергивает необходимое из <span class="wmi-link" show="http://ывов.рф">
     * 0 - Отображаемый урл
     * 1 - Реальный урл
     *
     * @return ArrayList<String[]>
     */
    public ArrayList<String[]> getAllLinksFromContent() {
        assertFalse("В ответе ручки " + this.getClass().getSimpleName() + " обнаружен тег ошибки",
                respAsString.contains("<error code="));
        ArrayList<String[]> links = new ArrayList<String[]>();

        NodeList nodes = documentConverter.byXpath("//span[@class='wmi-link']").asNodeList();


        // Заполняем список парами отображение-ссылка
        for (int i = 0; i < nodes.getLength(); i++) {
            links.add(new String[]{
                    nodes.item(i).getAttributes().getNamedItem("show").getTextContent(), // Отображ
                    nodes.item(i).getTextContent()});  // Реальный урл
        }

        nodes = documentConverter.byXpath("//content//a").asNodeList();

        // Заполняем список парами отображение-ссылка
        for (int i = 0; i < nodes.getLength(); i++) {
            links.add(new String[]{nodes.item(i).getTextContent(), // Отображ
                    nodes.item(i).getAttributes().getNamedItem("href").getTextContent()});  // Реальный урл
        }

        return links;
    }


    /**
     * Возвращает список из пар отображаемое значение-ссылка
     * Выдергивает необходимое из <span class="wmi-link" show="http://ывов.рф">
     * 0 - Отображаемый урл
     * 1 - Реальный урл
     *
     * @return String[]
     */
    public String[] getLinkFromContent() {
        String linkShow = documentConverter.byXpath("//span[@class='wmi-link']/@show").asString();
        String link = documentConverter.byXpath("//span[@class='wmi-link']/text()").asString();

        String linkShowDaria2 = documentConverter.byXpath("//content//a/text()").asString();
        String linkDaria2 = documentConverter.byXpath("//content//a/@href").asString();

        return new String[]{defaultIfEmpty(linkShow, linkShowDaria2), defaultIfEmpty(link, linkDaria2)};
    }

    public static Message msg(String mid) throws IOException {
        return api(Message.class).params(getMsg(mid).setXmlVersion(XMLVERSION_DARIA2));
    }

}
