package ru.yandex.autotests.innerpochta.wmi.core.oper;


import ch.ethz.ssh2.Connection;
import com.google.common.collect.Lists;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.jayway.jsonassert.JsonAssert;
import com.jayway.jsonassert.JsonAsserter;
import com.jayway.restassured.path.json.exception.JsonPathException;
import org.apache.http.Header;
import org.apache.http.StatusLine;
import org.apache.http.client.params.HttpClientParams;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicHeader;
import org.apache.log4j.Logger;
import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import ru.yandex.autotests.innerpochta.wmi.core.base.DocumentConverter;
import ru.yandex.autotests.innerpochta.wmi.core.base.RequestExecutor;
import ru.yandex.autotests.innerpochta.wmi.core.base.XpathConverter;
import ru.yandex.autotests.innerpochta.wmi.core.base.props.TvmProperties;
import ru.yandex.autotests.innerpochta.wmi.core.consts.Headers;
import ru.yandex.autotests.innerpochta.wmi.core.filter.AddHeaderFilter;
import ru.yandex.autotests.innerpochta.wmi.core.filter.Filter;
import ru.yandex.autotests.innerpochta.wmi.core.filter.log.LoggerFilterBuilder;
import ru.yandex.autotests.innerpochta.wmi.core.obj.EmptyObj;
import ru.yandex.autotests.innerpochta.wmi.core.obj.Obj;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;
import ru.yandex.autotests.innerpochta.wmicommon.WmiConsts;
import ru.yandex.autotests.plugins.testpers.html.common.Code;

import java.io.IOException;
import java.lang.reflect.Type;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.util.ArrayList;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import static com.jayway.restassured.path.json.JsonPath.from;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.jsx;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;
import static ru.yandex.autotests.innerpochta.wmi.core.consts.Headers.SERVICE_TICKET;
import static ru.yandex.autotests.innerpochta.wmi.core.consts.Headers.USER_TICKET;
import static ru.yandex.autotests.innerpochta.wmi.core.consts.Headers.WMI_TESTS;
import static ru.yandex.autotests.innerpochta.wmi.core.filter.CheckHeaderFilter.hasHeader;
import static ru.yandex.autotests.innerpochta.wmi.core.filter.CheckHeaderFilter.hasNoHeader;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.StatusCodeMatcher.hasStatusCode;
import static ru.yandex.autotests.innerpochta.wmicommon.WmiConsts.WmiErrorCodes.errByCode;

/**
 * <p>Created by IntelliJ IDEA.
 * <p>User: lanwen
 * <p>Date: 22.03.12
 * <p>Time: 14:29
 * <p>Базовый класс для всех операций
 */
@SuppressWarnings("unchecked")
public abstract class Oper<T extends Oper> {
    protected String description;
    protected String command;
    protected Obj obj;
    protected StatusLine statusLine;
    protected String host;

    /**
     * Логгер
     */
    protected Logger logger = Logger.getLogger(this.getClass());
    protected String respAsString;
    protected XpathConverter documentConverter;
    protected List<Filter> filters = new ArrayList<Filter>();


    protected Oper() {
        obj = new EmptyObj();
        documentConverter = new DocumentConverter();

        //AUTOTESTPERS-196
        if (!props().getCurrentRequestId().isEmpty()){
            header(Headers.REQUEST_ID, props().getCurrentRequestId());
        }
    }

    public String cmd() {
        return command;
    }

    public <V extends Oper<V>> V as(Class<V> clazz) {
        V instance = jsx(clazz);
        instance.setRespAsString(respAsString);
        instance.setStatusLine(statusLine);

        return instance;
    }


    /**
     * Клонирует текущий объект, чтобы при работе с несколькими респонсами,
     * они все не ссылались на одни и те ответы одних объектов
     * Копируются все поля, необходимые для выполнения запроса
     *
     * @return копия с пустыми полями ответа
     */
    public T clone() {
        T obj = (T) jsx(this.getClass());
        obj.description = this.description;
        obj.command = this.command;
        obj.host = this.host;
        obj.obj = this.obj;
        obj.filters = this.filters;
        return obj;
    }


    @SuppressWarnings({"unchecked"})
    public T params(Obj obj) {
        this.obj = obj;
        return (T) this;
    }

    /**
     * Удаляет назначенный объект свойств
     *
     * @return this
     */
    @SuppressWarnings({"unchecked"})
    public T noparams() {
        obj = new EmptyObj();
        return (T) this;
    }


    /**
     * Возвращает код ошибки, если есть
     *
     * @return - String test result
     */
    public String getErrorCode() {
        return documentConverter.byXpath("//error/@code").asString();
    }


    /**
     * проверяет соответствие кода ошибки
     */
    public T errorcode(WmiConsts.WmiErrorCodes expected) {
        Integer code = documentConverter.byXpath("//error/@code").asInteger();
        String hash = documentConverter.byXpath("//error/@hash").asString();

        assertThat(format("Код ошибки не совпадает с ожидаемым (хеш - %s)", hash), errByCode(code), is(expected));
        return (T) this;
    }

    /**
     * проверяет отсутствие кода ошибки
     */
    public T errorcodeShouldBeEmpty() {
        assertThat("В ответе присутствует код ошибки",
                documentConverter, new TypeSafeDiagnosingMatcher<XpathConverter>() {
                    @Override
                    protected boolean matchesSafely(XpathConverter item, Description mismatchDescription) {
                        String hash = documentConverter.byXpath("//error/@hash").asString();

                        if (isBlank(hash)) {
                            return true;
                        } else {
                            Integer code = documentConverter.byXpath("//error/@code").asInteger();
                            mismatchDescription.appendText("код ошибки ")
                                    .appendValue(errByCode(code)).appendText(", хеш - ").appendValue(hash);
                            return false;
                        }
                    }

                    @Override
                    public void describeTo(Description description) {
                        description.appendText(format("не должно быть ошибок в ответе команды %s", command));
                    }
                });
        return (T) this;
    }


    public T filters(Filter... filter) {
        filters.addAll(Arrays.asList(filter));
        return (T) this;
    }

    public T filters(int pos, Filter... filter) {
        filters.addAll(pos, Arrays.asList(filter));
        return (T) this;
    }

    public T noFilters() {
        filters.clear();
        return (T) this;
    }

    public T noFilter(Class<? extends Filter> clazz) {
        Lists.newArrayList(filters).stream()
                .filter(clazz::isInstance)
                .forEach(filters::remove);
        return (T) this;
    }


    public T log(LoggerFilterBuilder loggerFilterBuilder) {
        noFilter(loggerFilterBuilder.build().getClass());
        filters(loggerFilterBuilder.build());

        return (T) this;
    }

    public T headers(Header... headers) {
        filters(0, new AddHeaderFilter(headers));
        return (T) this;
    }

    public T noHeaders() {
        return noFilter(AddHeaderFilter.class);
    }

    public T header(String name, String value) {
        return headers(new BasicHeader(name, value));
    }

    public StatusLine statusLine() {
        return statusLine;
    }

    public void setStatusLine(StatusLine statusLine) {
        this.statusLine = statusLine;
    }


    public String toString() {
        return respAsString;
    }

    public T setHost(String host) {
        this.host = host;
        return (T) this;
    }

    public T setHost(URI uri) {
        this.host = uri.toString();
        return (T) this;
    }


    public String toFormattedString() {
        return Code.format(respAsString);
    }

    public void setRespAsString(String respAsString) {
        this.respAsString = respAsString;
        documentConverter.read(respAsString);
    }


    public Document toDocument() {
        return documentConverter.getConverted();
    }


    public T assertResponse(Matcher<String> matcher) {
        assertThat(respAsString, matcher);
        return (T) this;
    }


    public T assertResponse(String comment, Matcher<String> matcher) {
        assertThat(comment, respAsString, matcher);
        return (T) this;
    }

    public JsonAsserter withRespAsJson() {
        return JsonAssert.with(respAsString);
    }

    public String getStringFromJson(String path) {
        try {
            return from(respAsString).getString(path);
        } catch (JsonPathException e) {
            throw new AssertionError("Не удалось распарсить JSON", e);
        }
    }

    public List<Map<String, String>> getListFromJson(String path) {
        try {
            return from(respAsString).getList(path);
        } catch (JsonPathException e) {
            throw new AssertionError("Не удалось распарсить JSON", e);
        }
    }

    public int getIntFromJson(String path) {
        try {
            return from(respAsString).getInt(path);
        } catch (JsonPathException e) {
            throw new AssertionError("Не удалось распарсить JSON", e);
        }
    }

    public <K> K fromJson(Class<K> classOfK) {
        try {
            return new Gson().fromJson(respAsString, classOfK);
        } catch (JsonSyntaxException e) {
            throw new AssertionError("Не удалось распарсить JSON", e);
        }
    }

    public <K> K fromJson(Type typeOfK) {
        try {
            return new Gson().fromJson(respAsString, typeOfK);
        } catch (JsonSyntaxException e) {
            throw new AssertionError("Не удалось распарсить JSON", e);
        }
    }

    public T assertDocument(Matcher<Node> matcher) {
        assertThat(toDocument(), matcher);
        return (T) this;
    }


    public T assertDocument(String comment, Matcher<Node> matcher) {
        assertThat(comment, toDocument(), matcher);
        return (T) this;
    }

    public T statusCodeShouldBe(Integer code) {
        assertThat(this, hasStatusCode(code));
        return (T) this;
    }


    public T expectHeader(String header) {
        return filters(hasHeader(header));
    }


    public T expectNoHeader(String header) {
        return filters(hasNoHeader(header));
    }


    public T expectHeader(String header, Matcher<String> matcher) {
        return filters(hasHeader(header, matcher));
    }


    public T and() {
        return (T) this;
    }


    public T shouldBe() {
        return (T) this;
    }


    public T then() {
        return (T) this;
    }

    public T andReturn() {
        return (T) this;
    }


    public T withDebugPrint() {
        System.out.println(toFormattedString());
        return (T) this;
    }

    public String getRequest() {
        RequestExecutor requestExecutor = new RequestExecutor();
        return requestExecutor
                .cmd(command)
                .descr(description)
                .with(obj)
                .filters(filters)
                .host(host).getRequestAsString();
    }

    public Via post() {
        RequestExecutor requestExecutor = new RequestExecutor();

        requestExecutor
                .cmd(command)
                .descr(description)
                .with(obj)
                .filters(filters)
                .host(host)
                .requestType(RequestExecutor.RequestType.POST);

        return new Via(requestExecutor, this.clone());
    }

    public Via get() {
        RequestExecutor requestExecutor = new RequestExecutor();

        requestExecutor
                .cmd(command)
                .descr(description)
                .with(obj)
                .filters(filters)
                .host(host)
                .requestType(RequestExecutor.RequestType.GET);

        return new Via(requestExecutor, this.clone());
    }


    public Via head() {
        RequestExecutor requestExecutor = new RequestExecutor();

        requestExecutor
                .cmd(command)
                .descr(description)
                .with(obj)
                .filters(filters)
                .host(host)
                .requestType(RequestExecutor.RequestType.HEAD);

        return new Via(requestExecutor, this.clone());
    }


    public class Via {
        private RequestExecutor requestExecutor;
        private T resp;
        private boolean redirects = true;

        private Via(RequestExecutor requestExecutor, T resp) {
            this.requestExecutor = requestExecutor;
            this.resp = resp;
        }

        public Via redirects(boolean redirects) {
            this.redirects = redirects;
            return this;
        }

        public T via(DefaultHttpClient hc) {
            HttpClientParams.setRedirecting(hc.getParams(), redirects);
            try {
                requestExecutor.client(hc).execute(resp);
            } catch (SocketTimeoutException e) {
                throw new IllegalStateException(format("Таймаут выполнения запроса %s", requestExecutor.getRequestAsString()), e);
            } catch (IOException e) {
                throw new RuntimeException(format("Невозможно выполнить запрос %s", requestExecutor.getRequestAsString()), e);
            }
            return resp;
        }

        public T via(HttpClientManagerRule hc) {
            List<Filter> filters = Arrays.asList(
                    new AddHeaderFilter(new BasicHeader(SERVICE_TICKET, TvmProperties.props().ticketFor("hound"))),
                    new AddHeaderFilter(new BasicHeader(USER_TICKET, hc.account().userTicket())),
                    new AddHeaderFilter(new BasicHeader(WMI_TESTS, "oper"))
            );
            requestExecutor.filters(filters);
            return via(hc.authHC());
        }

        public T via(Connection connection) throws IOException {
            requestExecutor.client(new DefaultHttpClient()).execute(resp, connection);
            return resp;
        }

        public byte[] asBytes(DefaultHttpClient hc) throws IOException {
            return requestExecutor.client(hc).bytes(resp.log(LoggerFilterBuilder.log().body(false)));
        }
    }

}
