package ru.yandex.autotests.innerpochta.wmi.core.rules.acclock;

import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.ILock;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.junit.runner.Description;
import ru.yandex.autotests.innerpochta.objstruct.base.misc.Account;
import ru.yandex.autotests.innerpochta.wmi.core.base.anno.Credentials;
import ru.yandex.autotests.innerpochta.wmi.core.base.anno.IgnoreLock;
import ru.yandex.autotests.innerpochta.wmi.core.rules.TestWatcherWithExceptions;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.HazelcastClient;

import java.util.List;
import java.util.concurrent.TimeUnit;

import static ru.yandex.autotests.innerpochta.wmi.core.base.anno.Credentials.DEFAULT_LOGIN_GROUP;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 13.05.16
 * Time: 15:52
 */
public class AccLockRule extends TestWatcherWithExceptions {

    private final Logger logger = LogManager.getLogger(this.getClass());
    public static final long LOCK_TIME_SEC = 600;
    public static final long WAIT_TIME_SEC = 10;
    private HazelcastInstance hz;
    private ILock lock;
    private boolean lockAccounts = props().isLocsAccounts();
    private Account account = props().account(DEFAULT_LOGIN_GROUP);

    @Override
    protected void starting(Description description) throws InterruptedException {
        if (isLock(description)) {
            Credentials credentials = description.getTestClass().getAnnotation(Credentials.class);
            String loginGroup = credentials.loginGroup();
            hz = HazelcastClient.newHazelcastClient();
            List<Account> accounts = props().accounts(loginGroup);
            long start = System.currentTimeMillis();
            long end = start + TimeUnit.SECONDS.toMillis(LOCK_TIME_SEC);
            while (System.currentTimeMillis() < end) {
                for (Account account : accounts) {
                    this.lock = hz.getLock(lockname(account.getSelfEmail(), loginGroup));
                    this.account = account;
                    if (tryLock(lock)) {
                        return;
                    }
                }
                Thread.sleep(TimeUnit.SECONDS.toMillis(WAIT_TIME_SEC));
            }
        }

    }

    public Account getAccount() {
        return account;
    }

    /**
     * Лочим пользователя при условии:
     * - Есть аннотация @Credentials
     * - При этом в @Credentials loginGroup не дефолтный (не Default)
     * - Выставлена опция в конфиге lock.accounts = true
     * - Нет аннотации @IgnoreLock в тестовом классе
     *
     * @param description
     * @return
     */
    public boolean isLock(Description description) {
        return description.getTestClass().isAnnotationPresent(Credentials.class)
                && (!description.getTestClass().getAnnotation(Credentials.class).loginGroup().equals(DEFAULT_LOGIN_GROUP))
                && lockAccounts && !description.getTestClass().isAnnotationPresent(IgnoreLock.class);
    }

    @Step("Try lock account")
    public boolean tryLock(ILock lock) throws InterruptedException {
        logger.info("Try lock account: " + lock.getName());
        return lock.tryLock(0, TimeUnit.SECONDS, LOCK_TIME_SEC, TimeUnit.SECONDS);
    }

    @Step("Unlock user for test")
    public void unlock(ILock lock) {
        logger.info("Unlock account: " + lock.getName());
        lock.unlock();
    }

    @Step("Shoutdown hazelcast client")
    public void shutdown(HazelcastInstance hz) {
        logger.info("Shoutdown hazelcast client");
        hz.shutdown();
    }

    public String lockname(String selfEmail, String loginGroup) {
        return String.format("[wmi-tests]:[%s]:[%s]", loginGroup, selfEmail);
    }


    @Override
    protected void finished(Description description) {
        if (isLock(description)) {
            unlock(lock);
            shutdown(hz);
        }
    }
}
