package ru.yandex.autotests.innerpochta.wmi.core.rules.local;

import joptsimple.internal.Strings;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.log4j.LogManager;
import ru.yandex.autotests.innerpochta.beans.folderlist.Symbol;
import ru.yandex.autotests.innerpochta.wmi.core.obj.SettingsFolderDeleteObj;
import ru.yandex.autotests.innerpochta.wmi.core.obj.SettingsFolderSymbolObj;
import ru.yandex.autotests.innerpochta.wmi.core.oper.FolderList;
import ru.yandex.autotests.innerpochta.wmi.core.oper.SettingsFolderDeleteWithMsgs;
import ru.yandex.autotests.innerpochta.wmi.core.oper.SettingsFolderRemoveSymbol;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;
import ru.yandex.qatools.allure.annotations.Step;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static java.util.concurrent.TimeUnit.MINUTES;
import static org.hamcrest.MatcherAssert.assertThat;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.api;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.jsx;
import static ru.yandex.autotests.innerpochta.wmi.core.filter.log.LoggerFilterBuilder.log;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.IsNot.not;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.WaitForMatcherDecorator.withWaitFor;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.folders.IsThereFolderWithFidsMatcher.hasFoldersWithFids;

/**
 * Created with IntelliJ IDEA.
 * User: lanwen
 * Date: 30.03.13
 * Time: 19:14
 */
public class DeleteFoldersRule extends BeforeAfterOptionalRule<DeleteFoldersRule> {

    private DefaultHttpClient hc;
    private List<String> fidsToClean = new ArrayList<>();
    private List<String> fidsToExclude = new ArrayList<>();

    private static final long WAIT_TIME = MINUTES.toMillis(10);
    private boolean shouldWait = false;

    public static DeleteFoldersRule with(HttpClientManagerRule authClient) {
        return new DeleteFoldersRule(authClient.authHC());
    }

    private DeleteFoldersRule(DefaultHttpClient hc) {
        this.hc = hc;
    }

    public DeleteFoldersRule fid(String... fids) {
        fidsToClean.addAll(Arrays.asList(fids));
        return this;
    }

    public DeleteFoldersRule folderName(final String fname) {
        fid(
                api(FolderList.class)
                        .post().via(hc)
                        .getFolderId(fname)
        );

        return this;
    }

    public DeleteFoldersRule folderSymbol(final Symbol symbol) {
        final FolderList folders = api(FolderList.class).post().via(hc);
        final String name = folders.nameBySymbolOrNull(symbol);
        if (!Strings.isNullOrEmpty(name)) {
            fid(folders.getFolderId(name));
        }
        return this;
    }

    public DeleteFoldersRule all() {
        fidsToClean.addAll(api(FolderList.class)
            .log(log().onlyIfErrorOrXmlWithError())
            .post().via(hc)
            .getAllFolderIdsWithoutSystem());
        return this;
    }

    public DeleteFoldersRule exclude(String... fids) {
        fidsToExclude.addAll(Arrays.asList(fids));
        return this;
    }

    public DeleteFoldersRule shouldWait(boolean v) {
        this.shouldWait = v;
        return this;
    }

    @Step("[RULE]: Удаляем пользовательские папки")
    @Override
    public void call() {
        try {
            LogManager.getLogger(this.getClass()).trace("---------------CLEARING Folders---------------");

            SettingsFolderDeleteWithMsgs delOp = jsx(SettingsFolderDeleteWithMsgs.class);
            List<String> actualFids = new ArrayList<>(fidsToClean);
            actualFids.removeAll(fidsToExclude);
            boolean success = true;
            // Последовательно удаляем все
            for (String fid : actualFids) {
                success = success &&
                        jsx(SettingsFolderRemoveSymbol.class)
                        .log(log().onlyIfErrorOrXmlWithError())
                        .params(SettingsFolderSymbolObj.fid(fid)).post().via(hc).statusLine().getStatusCode() == 200;
                success = success &&
                        delOp.params(SettingsFolderDeleteObj.deleteOneFolder(fid))
                        .log(log().onlyIfErrorOrXmlWithError())
                        .post().via(hc).statusLine().getStatusCode() == 200;
            }

            if (success && shouldWait) {
                assertThat(hc, withWaitFor(not(hasFoldersWithFids(actualFids)), WAIT_TIME));
            }
        } catch (Exception e) {
            throw new RuntimeException("Невозможно выполнить удаление папок", e);
        }
    }
}
