package ru.yandex.autotests.innerpochta.wmi.core.rules.local;

import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.eclipse.jetty.http.HttpSchemes;
import org.junit.rules.ExternalResource;
import ru.yandex.autotests.innerpochta.wmi.core.rules.SSHRule;

import javax.ws.rs.core.UriBuilder;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.URI;
import java.net.UnknownHostException;

import static java.lang.String.format;
import static java.net.InetAddress.getLocalHost;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 07.08.14
 * Time: 17:10
 */
public class SshRemotePortForwardingRule extends ExternalResource {

    public static final URI DEFAULT_BIND_ADDRESS_ON_REMOTE = URI.create("ssh://127.0.0.1:12345");

    private SSHRule sshRule = new SSHRule(Logger.getLogger(this.getClass()));

    private Logger logger;

    private String hostOnRemote;
    private int portOnRemote;
    private String hostToForward;
    private int portToForward;


    private SshRemotePortForwardingRule(URI uri) {
        sshRule.serverUrl(uri);
        logger = LogManager.getLogger(this.getClass());
        whenRemoteConnectsTo(DEFAULT_BIND_ADDRESS_ON_REMOTE).forwardToLocal();
    }

    public static int localPortForMocking() {
        try (ServerSocket socket = new ServerSocket(0)) {
            socket.setReuseAddress(true);
            return socket.getLocalPort();
        } catch (IOException e) {
            throw new RuntimeException("Не удалось найти свободный TCP/IP порт", e);
        }
    }

    @Override
    protected void before() throws Throwable {
        sshRule.auth();
        logger.info(format("Try to create port forwarding: `ssh %s -l %s -f -N -R %s:%s:%s:%s`",
                sshRule.conn().getHostname(), "robot-aqua-testpers",
                hostOnRemote, portOnRemote, hostToForward, portToForward
        ));

        sshRule.conn().requestRemotePortForwarding(hostOnRemote, portOnRemote, hostToForward, portToForward);
    }

    @Override
    protected void after() {
        try {
            sshRule.conn().cancelRemotePortForwarding(portOnRemote);
        } catch (IOException e) {
            throw new RuntimeException("Can't cancel remote port forwarding", e);
        } finally {
            sshRule.close();
        }
    }

    public SshRemotePortForwardingRule forwardToLocal() {
        try {
            hostToForward = getLocalHost().getHostAddress();
        } catch (UnknownHostException e) {
            throw new RuntimeException("Can't get local host", e);
        }
        return this;
    }


    public SshRemotePortForwardingRule whenRemoteConnectsTo(URI uri) {
        hostOnRemote = uri.getHost();
        portOnRemote = uri.getPort();
        return this;
    }

    public SshRemotePortForwardingRule whenRemoteUsesPort(int port) {
        portOnRemote = port;
        return this;
    }

    public SshRemotePortForwardingRule withForwardToPort(int port) {
        portToForward = port;
        return this;
    }

    public static SshRemotePortForwardingRule onRemoteHost(URI uri) {
        return new SshRemotePortForwardingRule(uri);
    }

    public URI onRemoteUri() {
        return UriBuilder.fromPath("").host(hostOnRemote).port(portOnRemote).scheme(HttpSchemes.HTTP).build();
    }
}