package ru.yandex.autotests.innerpochta.wmi.core.rules.mock;

import org.apache.log4j.LogManager;
import org.apache.mina.core.filterchain.DefaultIoFilterChainBuilder;
import org.apache.mina.core.service.IoAcceptor;
import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.filter.codec.textline.TextLineCodecFactory;
import org.apache.mina.filter.logging.LoggingFilter;
import org.apache.mina.transport.socket.nio.NioSocketAcceptor;
import org.junit.rules.ExternalResource;

import java.io.Closeable;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.charset.Charset;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 28.04.15
 * Time: 19:47
 */
public class MinaRule extends ExternalResource implements Closeable {

    public static final String CODEC = "codec";

    private final org.apache.log4j.Logger logger = LogManager.getLogger(this.getClass());

    private InetSocketAddress socket;
    private IoAcceptor acceptor;

    public MinaRule(int port) {
        this.socket = new InetSocketAddress(port);
        acceptor = new NioSocketAcceptor();
    }

    @Override
    public void close() {
        logger.info(String.format("Stopping mina server on %s...", socket.getPort()));
        acceptor.unbind(socket);
        acceptor.dispose();
    }

    public void open() throws IOException {
        logger.info(String.format("Starting mina server on %s... ", socket.getPort()));
        acceptor.bind(socket);
    }

    public void configureDefaultMinaServer() throws IOException {
        acceptor.getFilterChain().addFirst("logger", new LoggingFilter());
        withDefaultCodec();
        withDefaultHandler();

    }

    @Override
    protected void before() throws Throwable {
        configureDefaultMinaServer();
        open();
    }

    public MinaRule withCodec(ProtocolCodecFilter codec) {
        acceptor.getFilterChain().addLast(CODEC, codec);
        return this;
    }

    private void withDefaultCodec() {
        if (!acceptor.getFilterChain().contains(CODEC)) {
            withCodec(new ProtocolCodecFilter(new TextLineCodecFactory(Charset.defaultCharset(), System.lineSeparator(),
                    System.lineSeparator())));
        }
    }

    /**
     * Handler обязателен к установке
     */
    private void withDefaultHandler() {
        if (acceptor.getHandler() == null) {
            acceptor.setHandler(new IoHandlerAdapter());
        }
    }

    /**
     * Для установки хэндлера нужно перезапускать мину
     *
     * @param handler
     * @return
     */
    public MinaRule withHandler(IoHandlerAdapter handler) {
        if (acceptor.isActive()) {
            logger.info(String.format("Restarting mina server on %s with new handler...", socket.getAddress().getHostAddress()));
            DefaultIoFilterChainBuilder filterChain = acceptor.getFilterChain();
            close();
            acceptor = new NioSocketAcceptor();
            acceptor.setFilterChainBuilder(filterChain);
            acceptor.setHandler(handler);
            try {
                open();
            } catch (IOException e) {
                throw new RuntimeException("Не удалось рестартануть сервер mina", e);
            }
            return this;
        }
        acceptor.setHandler(handler);
        return this;
    }

    @Override
    protected void after() {
        close();
    }
}
