package ru.yandex.autotests.innerpochta.wmi.core.rules.mock;

import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.eclipse.jetty.http.HttpSchemes;
import org.junit.rules.ExternalResource;
import ru.yandex.autotests.innerpochta.wmi.core.rules.SSHRule;

import javax.ws.rs.core.UriBuilder;
import java.io.IOException;
import java.net.*;
import java.util.ArrayList;
import java.util.List;

import static java.lang.String.format;
import static java.net.InetAddress.getLocalHost;
import static org.apache.commons.lang3.Validate.isTrue;
import static org.apache.commons.lang3.Validate.notNull;
import static ru.yandex.autotests.innerpochta.util.ssh.SSHCommands.executeCommAndResturnResultAsString;
import static ru.yandex.autotests.innerpochta.wmi.core.rules.mock.Service.WITHOUT_PORT;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 07.08.14
 * Time: 17:10
 */
public class MockRule extends ExternalResource {

    public static final URI IPV4_LOCALHOST = URI.create("ssh://127.0.0.1:1234");
    public static final URI IPV6_LOCALHOST = URI.create("ssh://[::1]:1234");

    private SSHRule sshRule = new SSHRule(Logger.getLogger(this.getClass()));

    private Logger logger;

    private String hostOnRemote;
    private int portOnRemote;
    private String hostToForward;
    private int portToForward;

    private List<Service> services = new ArrayList<>();

    private MockRule(URI uri, URI bindAddressOnRemote) {
        sshRule.serverUrl(uri);
        logger = LogManager.getLogger(this.getClass());
        whenRemoteConnectsTo(bindAddressOnRemote).forwardToLocal();
        portOnRemote = -1;
    }

    public static int localPortForMocking() {
        try (ServerSocket socket = new ServerSocket(0)) {
            socket.setReuseAddress(true);
            return socket.getLocalPort();
        } catch (IOException e) {
            throw new RuntimeException("Не удалось найти свободный TCP/IP порт", e);
        }
    }

    /**
     * Получаем свободный порт на машине
     * @return
     * @throws IOException
     */
    private int remotePortForMocking() throws IOException {
        int port = IPV4_LOCALHOST.getPort();
        for (int i = 0; i < 5; i++) {
            // Ports from 50001 up to 65536
            port = (int) (Math.random() * 15535 + 50001);
            String netstat = String.format("sudo netstat -anp  2>/dev/null | grep -q \":%d \" && echo \"no\" || echo \"ok\"", port);
            String result = executeCommAndResturnResultAsString(sshRule.conn(), netstat, LogManager.getLogger(MockRule.class));
            if (result.contains("ok")) {
                return port;
            }
        }

        return port;
    }

    @Override
    protected void before() throws Throwable {
        sshRule.auth();
        //получаем ip
        String ip = InetAddress.getLocalHost().getHostAddress();
        logger.info("Current IP: " + ip);
        for (Service service : services) {
            //прописываем правило в iptables
            IptablesHelper appendRule = new IptablesHelper().nat().append().host(service.getHost());
            if (service.getPort() != WITHOUT_PORT) {
                appendRule.port(service.getPort());
            }

            if (portOnRemote == WITHOUT_PORT) {
                portOnRemote = remotePortForMocking();
            }

            appendRule.toDestination(hostOnRemote, portOnRemote);

            if (isNotLocalhost(service.getHost())) {
                //сначала удаляем из хостов, если есть
                removeFromHosts(service.getHost());
                //настраиваем локальный dns для мокнутого сервера
                logger.info(String.format("Add to hosts: %s %s", service.getHost(), ip));
                String echo = String.format("sudo -- sh -c \"echo '%s %s' >> /etc/hosts\"", ip, service.getHost());
                executeCommAndResturnResultAsString(sshRule.conn(), echo, logger);
            }

            executeCommAndResturnResultAsString(sshRule.conn(), appendRule.getCommand(), logger);
            executeCommAndResturnResultAsString(sshRule.conn(), appendRule.getCommand6(), logger);
        }
        //создаем ssh-туннель
        logger.info(format("Try to create port forwarding: `ssh %s -l %s -f -N -R %s:%s:%s:%s`",
                sshRule.conn().getHostname(), "robot-aqua-testpers",
                hostOnRemote, portOnRemote, hostToForward, portToForward
        ));
        sshRule.conn().requestRemotePortForwarding(hostOnRemote, portOnRemote, hostToForward, portToForward);
    }

    @Override
    protected void after() {
        try {
            sshRule.conn().cancelRemotePortForwarding(portOnRemote);
        } catch (IOException e) {
            throw new RuntimeException("Can't cancel remote port forwarding", e);
        }
        try {
            for (Service service : services) {
                //удаляем правило в iptables
                IptablesHelper deleteRule = new IptablesHelper().nat().delete().host(service.getHost());
                if (service.getPort() != WITHOUT_PORT) {
                    deleteRule.port(service.getPort());
                }
                deleteRule.toDestination(hostOnRemote, portOnRemote);
                executeCommAndResturnResultAsString(sshRule.conn(), deleteRule.getCommand(), logger);
                executeCommAndResturnResultAsString(sshRule.conn(), deleteRule.getCommand6(), logger);

                if (isNotLocalhost(service.getHost())) {
                    //удаляем из hosts добавленные настройки
                    removeFromHosts(service.getHost());
                }
            }
        } catch (IOException e) {
            throw new RuntimeException("Can't clear iptables", e);
        } finally {
            //закрываем ssh-туннель
            sshRule.close();
        }
    }

    private void removeFromHosts(String host) throws IOException {
        logger.info(String.format("Remove from /etc/hosts: %s", host));
        String sed = String.format("sudo sed -ie \"\\|%s\\$|d\" /etc/hosts", host);
        executeCommAndResturnResultAsString(sshRule.conn(), sed, logger);
    }

    private MockRule forwardToLocal() {
        try {
            hostToForward = getLocalHost().getHostAddress();
        } catch (UnknownHostException e) {
            throw new RuntimeException("Can't get local host", e);
        }
        return this;
    }

    public MockRule mock(String host) {
        services.add(new Service().setHost(notNull(host, "config error - host can't be null")));
        isTrue(isNotLocalhost(host), "Error: port parameter is not necessary for localhost");
        return this;
    }

    public MockRule ssh(SSHRule sshRule) {
        this.sshRule = sshRule;
        return this;
    }

    public MockRule mock(String host, int port) {
        services.add(new Service().setHost(host).setPort(port));
        return this;
    }

    public MockRule mock(int port) {
        services.add(new Service().setPort(port));
        return this;
    }

    private Boolean isNotLocalhost(String host) {
        return !isLocalhost(host);
    }

    private Boolean isLocalhost(String host) {
        return (host.equals(Service.LOCALHOST) || (host.equals("localhost")));
    }

    public MockRule whenRemoteConnectsTo(URI uri) {
        hostOnRemote = uri.getHost();
        portOnRemote = uri.getPort();
        return this;
    }

    public MockRule whenRemoteUsesPort(int port) {
        portOnRemote = port;
        return this;
    }

    public MockRule withForwardToPort(int port) {
        portToForward = port;
        return this;
    }

    public static MockRule onLocalhost(URI uri) {
        return new MockRule(uri, IPV4_LOCALHOST);
    }

    public static MockRule onRemoteIPV6Host(URI uri) {
        return new MockRule(uri, IPV6_LOCALHOST);
    }

    public URI onRemoteUri() {
        return UriBuilder.fromPath("").host(hostOnRemote).port(portOnRemote).scheme(HttpSchemes.HTTP).build();
    }
}
