package ru.yandex.autotests.innerpochta.wmi.core.sendbernar;


import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.jayway.restassured.builder.ResponseSpecBuilder;
import com.jayway.restassured.specification.ResponseSpecification;
import org.apache.http.HttpStatus;
import com.google.gson.reflect.TypeToken;
import org.json.JSONObject;
import org.json.JSONException;
import org.jsoup.Jsoup;
import ru.yandex.autotests.innerpochta.beans.mbody.AddressesResult;
import ru.yandex.autotests.innerpochta.beans.mbody.Attachment;
import ru.yandex.autotests.innerpochta.beans.mbody.Mbody;
import ru.yandex.autotests.innerpochta.beans.mdoby.Flag;
import ru.yandex.autotests.innerpochta.wmi.core.oper.hound.FilterSearchCommand;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;

import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static java.util.function.Function.identity;
import static ru.yandex.autotests.innerpochta.wmi.core.api.CommonApiSettings.shouldBe;
import static ru.yandex.autotests.innerpochta.wmi.core.api.MbodyApi.apiMbody;
import static ru.yandex.autotests.innerpochta.wmi.core.obj.hound.FilterSearchObj.empty;

public class Message {
    private String mid;
    private HttpClientManagerRule rule;
    private Mbody mbodyResp;
    private String headersResp;
    private String sourceResp;
    private FilterSearchCommand filterSearchResp;

    private static ResponseSpecification ok200() {
        return new ResponseSpecBuilder()
                .expectStatusCode(HttpStatus.SC_OK)
                .build();
    }

    private Mbody mbody() {
        if (mbodyResp == null) {
            mbodyResp = apiMbody(rule.account().userTicket())
                    .message()
                    .withUid(getUid())
                    .withMid(mid)
                    .withFlags(Flag.OUTPUT_AS_CDATA.toString())
                    .get(shouldBe(ok200()))
                    .as(Mbody.class);
        }

        return mbodyResp;
    }

    private String getUid() {
        return rule.account().uid();
    }

    private String headers() {
        if (headersResp == null) {
            headersResp = apiMbody(rule.account().userTicket()).headers()
                    .withMid(mid)
                    .withUid(getUid())
                    .get(identity())
                    .peek()
                    .asString();
        }

        return headersResp;
    }

    private String messageSource() throws Exception {
        if (sourceResp == null) {
            String resp = apiMbody(rule.account().userTicket()).messageSource()
                    .withMid(mid)
                    .withUid(getUid())
                    .get(shouldBe(ok200()))
                    .peek()
                    .asString();


            JSONObject respJson = new JSONObject(resp);
            sourceResp = new String(Base64.getDecoder().decode(respJson.getString("text")));
        }

        return sourceResp;
    }

    private FilterSearchCommand filterSearch() {
        if (filterSearchResp == null) {
            filterSearchResp = FilterSearchCommand.filterSearch(empty().setUid(getUid()).setMids(mid)).get().via(rule);
        }

        return filterSearchResp;
    }

    public Message(String mid, HttpClientManagerRule rule) {
        this.mid = mid;
        this.rule = rule;
        this.mbodyResp = null;
        this.filterSearchResp = null;
    }

    public String getHeader(String header) {
        header = header.toLowerCase();
        JsonParser parser = new JsonParser();
        JsonElement headersArray = parser.parse(headers())
                .getAsJsonObject().get(mid)
                .getAsJsonObject().get(header);

        if (headersArray == null) {
            return null;
        } else {
            return headersArray.getAsJsonArray().get(0).getAsString();
        }
    }

    public List<String> getHeaders(String header) {
        header = header.toLowerCase();
        JsonParser parser = new JsonParser();
        JsonElement headers = parser.parse(headers())
                .getAsJsonObject().get(mid)
                .getAsJsonObject().get(header)
                .getAsJsonArray();

        return new Gson().fromJson(headers, new TypeToken<List<String>>(){}.getType());
    }

    public List<Attachment> getAttachments() {
        return mbody().getAttachments();
    }

    public String getAttachHidByName(String name) {
        List<Attachment> attachments = mbody().getAttachments();

        Optional<Attachment> att = attachments
                .stream()
                .filter(entry -> entry
                                .getBinaryTransformerResult()
                                .getTypeInfo()
                                .getName()
                                .equals(name))
                .findFirst();

        return att.isPresent() ? att.get().getBinaryTransformerResult().getHid() : null;
    }

    public String noReplyNotification() {
        return mbody()
                .getInfo()
                .getNoReplyResult()
                .getNotification();
    }

    public String firstline() {
        return filterSearch().firstline();
    }

    public String subject() {
        return filterSearch().subject();
    }

    public String content() {
        return mbody()
                .getBodies().get(0)
                .getTransformerResult()
                .getTextTransformerResult()
                .getContent();
    }

    public String sourceContent() throws Exception {
        String messageSourceText = messageSource();

        String separatorString = "\r\n\r\n";
        int contentStartIndex = messageSourceText.indexOf(separatorString);
        if (0 <= contentStartIndex) {
            return messageSourceText.substring(contentStartIndex + separatorString.length());
        }

        return null;
    }

    public String text() {
        return Jsoup.parse(content()).text();
    }

    public String fromEmail() {
        Map<String, String> from = filterSearch().from();

        return from.get("local")+"@"+from.get("domain");
    }

    public String fromName() {
        Map<String, String> from = filterSearch().from();

        return from.get("displayName");
    }

    public String toEmail() {
        Map<String, String> to = filterSearch().to().get(0);

        return to.get("local")+"@"+to.get("domain");
    }

    public List<String> toEmailList() {
        return filterSearch()
                .to()
                .stream()
                .map(email -> String.format("%s@%s", email.get("local"), email.get("domain")).toLowerCase())
                .collect(Collectors.toList());
    }

    public String ccEmail() {
        List<Map<String, String>> ccList = filterSearch().cc();

        return ccList.isEmpty() ? "" : ccList.get(0).get("local")+"@"+ccList.get(0).get("domain");
    }

    public String bccEmail() {
        Optional<String> bcc = mbody()
                .getInfo()
                .getAddressesResult()
                .stream()
                .filter(email -> email
                                    .getDirection()
                                    .equals("bcc"))
                .map(AddressesResult::getEmail)
                .findFirst();

        return bcc.isPresent() ? bcc.get().toLowerCase() : "";
    }

    public String messageId() {
        return mbody().getInfo().getMessageId();
    }

    public void exists() {
        mbody();
    }
}
