package ru.yandex.autotests.innerpochta.wmi.core.utils;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.fluent.Executor;
import org.apache.http.client.fluent.Request;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.util.EntityUtils;
import org.apache.log4j.LogManager;
import ru.yandex.autotests.innerpochta.util.LogToFileUtils;
import ru.yandex.autotests.innerpochta.wmi.core.obj.MessageObj;
import ru.yandex.autotests.innerpochta.wmi.core.obj.MessagePartObj;
import ru.yandex.autotests.innerpochta.wmi.core.oper.MailBoxList;
import ru.yandex.autotests.innerpochta.wmi.core.oper.Message;
import ru.yandex.autotests.innerpochta.wmi.core.oper.MessagePart;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;
import ru.yandex.autotests.innerpochta.wmicommon.Util;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.net.URL;
import java.util.List;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import static com.google.common.io.Files.asByteSink;
import static com.google.common.io.Resources.asByteSource;
import static com.google.common.io.Resources.getResource;
import static java.lang.String.format;
import static org.apache.commons.lang.RandomStringUtils.randomAlphabetic;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.Matchers.isEmptyOrNullString;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.MatcherAssert.assertThat;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.api;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.jsx;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.FileCompareMatcher.hasSameMd5As;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.BufferedImageMatcher.hasSameTypeAndSize;
import static ru.yandex.autotests.innerpochta.wmi.core.obj.MessagePartObj.getEmptyObj;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 9/13/13
 * Time: 9:44 PM
 */
public class AttachUtils {

    private static HttpClientManagerRule authClient;

    public static void setAuthClient(HttpClientManagerRule authClient) {
        AttachUtils.authClient = authClient;
    }

    /**
     * Создает в папке отчетов файл с нужным именем и копирует туда
     * файл по урлу
     *
     * @param url  - прямой урл для скачивания файла
     * @param name - имя файла - будет добавлено после префикса и случайной части
     * @return File - объект сохраненного файла
     * @throws java.io.IOException *
     */
    public static File downloadFile(String url, final String name, DefaultHttpClient hc) throws IOException {
        LogManager.getLogger("FILE Downloader").info("Downloading file from url " + url);

        // Мавену почему-то не нравится использование функции с таймаутами

        return Executor.newInstance(hc)
                .execute(Request.Get(url))
                .handleResponse(new ResponseHandler<File>() {
                    @Override
                    public File handleResponse(HttpResponse response) throws ClientProtocolException, IOException {
                        System.out.println(response.toString());
                        File file = LogToFileUtils.getLogFileWithoutLogEvent("dowloaded_", name);
                        LogManager.getLogger(LogToFileUtils.class).info("See log in [[a href=" + file.getAbsolutePath() + "]]this file[[/a]]");
                        FileUtils.writeByteArrayToFile(file, EntityUtils.toByteArray(response.getEntity()));
                        return file;
                    }
                });
    }

    /**
     * Скачивает содержимое файла в byte[] по урлу
     *
     * @param url - прямой урл для скачивания файла
     * @return byte[] - содержимое файла
     * @throws java.io.IOException *
     */
    public static byte[] downloadFileBody(String url, DefaultHttpClient hc) throws IOException {
        LogManager.getLogger("FILE Downloader").info("Downloading file from url " + url);

        return Executor.newInstance(hc)
                .execute(Request.Get(url))
                .handleResponse(new ResponseHandler<byte[]>() {
                    @Override
                    public byte[] handleResponse(HttpResponse response) throws ClientProtocolException, IOException {
                        byte[] result = EntityUtils.toByteArray(response.getEntity());
                        return result;
                    }
                });
    }

    /**
     * Вычисляет хеш байтового массива
     *
     * @param array - сам массив
     * @return byte[] - хеш
     * @throws java.security.NoSuchAlgorithmException *
     */
    public static byte[] byteArrayMd5(byte[] array) throws NoSuchAlgorithmException {
        MessageDigest md = MessageDigest.getInstance("MD5");
        md.update(array);
        return md.digest();
    }

    /**
     * Проверка соответствия аттачей скачанных залитым
     *
     * @param attach - аттач для проверки
     * @throws Exception *
     */
    public static void attachInMessageShouldBeSameAs(File attach, String subj, DefaultHttpClient hc) throws Exception {
        // Получаем мид
        String mid = jsx(MailBoxList.class)
                .post().via(hc).getMidOfMessage(subj);
        // Получаем вывод письма
        Message resp = api(Message.class)
                .params(MessageObj.getMsgWithContentFlag(mid))
                .post().via(hc);
        String hid = resp.getHidOfAttach(attach.getName());
        String url = api(MessagePart.class)
                .params(MessagePartObj.getUrlObj(mid, hid, attach.getName()))
                .post().via(hc)
                .toString();
        File deliveredAttach = downloadFile(url, attach.getName(), hc);
        assertThat("MD5 хэши файлов не совпали ", deliveredAttach, hasSameMd5As(attach));
    }

    /**
     * Проверка соответствия аттачей скачанных залитым
     *
     * @param attach - аттач для проверки
     * @throws Exception *
     */
    public static void attachsInMessageShouldBeSameAs(File attach, String subj, DefaultHttpClient hc) throws Exception {
        // Получаем мид
        String mid = jsx(MailBoxList.class)
                .post().via(hc).getMidOfMessage(subj);
        // Получаем вывод письма
        Message resp = api(Message.class)
                .params(MessageObj.getMsgWithContentFlag(mid))
                .post().via(hc);
        List<String> hids = resp.getAllHids();
        // удаляем тело письма
        hids.remove("1.1");

        for (String hid : hids) {
            String url = api(MessagePart.class)
                    .params(MessagePartObj.getUrlObj(mid, hid, attach.getName()))
                    .post().via(hc)
                    .toString();
            File deliveredAttach = downloadFile(url, attach.getName(), hc);
            assertThat("MD5 хэши файлов не совпали " + hid, deliveredAttach, hasSameMd5As(attach));
        }
    }

    /**
     * Проверяет количество аттачей в письме
     *
     * @param subj  - тема письма
     * @param count - ожидаемое количество аттачей
     * @param hc
     * @throws Exception
     */

    public static void attachsInMessageShouldBeCountAs(String subj, int count, DefaultHttpClient hc) throws Exception {

        String mid = jsx(MailBoxList.class)
                .post().via(hc).getMidOfMessage(subj);

        int countAttach = api(Message.class)
                .params(MessageObj.getMsgWithContentFlag(mid))
                .post().via(hc).getAllHids().size() - 1;

        assertThat("Количество аттачей отличается ", countAttach, equalTo(count));
    }

    public static File genFile(long sizeBytes) throws IOException {
        LogManager.getLogger(AttachUtils.class).trace(format("Generate file with size [%s]", sizeBytes));
        File randomFile = File.createTempFile(randomAlphabetic(10), ".tmp");
        int buffer = 1024;
        RandomAccessFile file = new RandomAccessFile(randomFile, "rw");
        for (int i = 0; i < sizeBytes; i += buffer) {
            file.writeBytes(RandomStringUtils.random(buffer));
        }
        file.setLength(sizeBytes);
        return randomFile;
    }

    public static File genFile(String filename) throws IOException {
        LogManager.getLogger(AttachUtils.class).trace(format("Generate empty temporary file with filename [%s]", filename));
        File tmpFile = new File(System.getProperty("java.io.tmpdir", "/tmp/"), filename);
        tmpFile.createNewFile();
        tmpFile.deleteOnExit();
        return tmpFile;
    }

    public static String fromClasspath(String path) throws IOException {
        try(final InputStream stream = AttachUtils.class.getClassLoader().getResourceAsStream(path)) {
            assertThat(stream, notNullValue());
            String result = IOUtils.toString(stream, Charsets.UTF_8);
            assertThat(result, notNullValue());
            return result;
        }
    }

    /**
     * @param path путь в директории resources
     * @return
     * @throws IOException
     */
    public static File getFileFromPath(String path) throws IOException {
        String fileToCompare = Util.getRandomString();
        File previewImage = File.createTempFile(fileToCompare, null);
        asByteSink(previewImage).write(asByteSource(getResource(path)).read());
        return previewImage;
    }

    public static File getFileFromPath(String path, String suffix) throws IOException {
        String fileToCompare = Util.getRandomString();
        File previewImage = File.createTempFile(fileToCompare, suffix);
        asByteSink(previewImage).write(asByteSource(getResource(path)).read());
        return previewImage;
    }
}
