package ru.yandex.autotests.innerpochta.wmi.core.utils;

import com.jayway.restassured.response.Cookies;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.log4j.LogManager;
import ru.yandex.autotests.innerpochta.beans.yplatform.Lid;
import ru.yandex.autotests.innerpochta.objstruct.base.misc.Account;
import ru.yandex.autotests.innerpochta.wmi.core.obj.MailBoxListObj;
import ru.yandex.autotests.innerpochta.wmi.core.oper.MailBoxList;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;

import javax.ws.rs.core.UriBuilder;
import java.io.IOException;
import java.io.InputStream;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.jayway.restassured.RestAssured.given;
import static java.util.Arrays.asList;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.jsx;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;
import static ru.yandex.autotests.innerpochta.wmi.core.filter.log.LoggerFilterBuilder.log;
import static ru.yandex.autotests.innerpochta.wmi.core.obj.MailBoxListObj.inFid;
import static ru.yandex.autotests.innerpochta.wmi.core.oper.newapi.LabelsCommand.labels;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 9/13/13
 * Time: 9:48 PM
 */
public class CommonUtils {

    private static HttpClientManagerRule authClient;

    public static void setAuthClient(HttpClientManagerRule authClient) {
        CommonUtils.authClient = authClient;
    }

    /**
     * Возвращает список из всех мидов этой папки, на всех страницах
     *
     * @param fid - фид папки, где ищем миды
     * @return - Список мидов-строк
     * @throws java.io.IOException
     */
    public static List<String> getMidsOfAllMsgsInFolder(String fid, DefaultHttpClient hc) throws IOException {
        HashSet<String> set = new HashSet<>();
        Integer page = 1;

        List<String> midsOnPage;
        do {
            // получаем список мидов на странице и добавляем его в общий список
            midsOnPage = jsx(MailBoxList.class)
                    .log(log().nothing())
                    .params(inFid(fid).setPageNumber(page++))
                    .post().via(hc).getMidsOfMessagesInFolder();
            set.addAll(midsOnPage);
        } while (!midsOnPage.isEmpty()); // пока список мидов со страницы не пуст
        return new ArrayList<>(set);
    }

    /**
     * Возвращает первый найденный мид сообщения с указанной темой
     *
     * @param fid  - фид папки, где ищем мид
     * @param subj - тема письма
     * @return - мид
     * @throws IOException
     */
    public static String getFirstMidAllPagesInFolder(String subj, String fid, DefaultHttpClient hc)
            throws IOException {
        Integer page = 1;
        MailBoxListObj mboxObj = inFid(fid)
                .setPageNumber(page);
        MailBoxList mboxOp = jsx(MailBoxList.class).params(mboxObj);

        List<String> midsOnPage;
        String mid;
        do {
            MailBoxList resp = mboxOp.post().via(hc);
            mid = resp.getMidOfMessage(subj);

            if (mid != null) {
                return mid;
            }
            // получаем список мидов на странице и добавляетм его в общий список
            midsOnPage = resp.getMidsOfMessagesInFolder();
            // инкрементим страницу и назначаем ее объекту
            mboxObj.setPageNumber(++page);
        } while (midsOnPage.size() != 0); // пока список мидов со страницы не пуст

        LogManager.getLogger("SearcherOfMidAllPages")
                .error("Письмо с темой " + subj + " в папке " + fid + " не найдено");
        return null;
    }

    public static String getFirstMidAllPagesInFolder(String subj, String fid) throws IOException {
        return getFirstMidAllPagesInFolder(subj, fid, authClient.authHC());
    }

    public static Cookies authWith(List<Account> accounts) {
        Cookies cookies = new Cookies();
        for (Account next : accounts) {
            cookies = given().redirects().follow(true)
                    .cookies(cookies)
                    .baseUri(props().passportHost().toString())
                    .queryParam("mode", "auth")
                    .formParam("login", next.getLogin())
                    .formParam("passwd", next.getPassword())
                    .post("/passport").detailedCookies();
        }
        return cookies;
    }


    public static Cookies authWith(Account... accounts) {
        return authWith(asList(accounts));
    }

    public static String getResource(Class clas, String path) throws IOException {
        try(final InputStream stream = clas.getResourceAsStream(path)) {
            return IOUtils.toString(stream);
        }
    }

    public static String getUriPattern(String request, String uid) throws UnknownHostException {
        return StringUtils.replace(UriBuilder.fromPath(request)
                .queryParam("uid", uid)
                .toString(), "?", "\\?");
    }

    public static Optional<String> getLidBySymbolTitle(String symbolTitle, DefaultHttpClient hc) {
        Map<String, Lid> labels = labels().get().via(hc).statusCodeShouldBe(HttpStatus.SC_OK).resp();
        return labels.entrySet().stream()
                .filter((entry) ->
                        entry.getValue().getSymbolicName().getTitle().equals(symbolTitle)
                ).map(Map.Entry::getKey)
                .findFirst();
    }
}
