package ru.yandex.autotests.innerpochta.wmi.core.utils;

import com.google.common.base.Joiner;
import lombok.val;
import org.w3c.dom.Document;
import ru.yandex.autotests.innerpochta.wmi.core.oper.FolderList;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;
import ru.yandex.autotests.innerpochta.wmicommon.Util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.api;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.DocumentCompareMatcher.equalToDoc;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.folders.IsThereSubfolderMatcher.hasSubfolder;
import static ru.yandex.autotests.innerpochta.wmi.core.oper.SettingsFolderCreate.newChildFolder;
import static ru.yandex.autotests.innerpochta.wmi.core.oper.SettingsFolderSetOptions.firstPlaceSortOptions;
import static ru.yandex.autotests.innerpochta.wmi.core.oper.SettingsFolderSetOptions.setSortOptions;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 04.10.13
 * Time: 19:23
 */
public class FolderUtils {

    private static HttpClientManagerRule authClient;

    public static void setAuthClient(HttpClientManagerRule authClient) {
        FolderUtils.authClient = authClient;
    }

    /**
     * Создает подпапку в папке с известным именем
     *
     * @param parentFolderName - ПОЛНОЕ имя родительской папки (т.е. со всеми предками через '|')
     * @param childFolderName  - имя вложенной папки
     * @throws Exception *
     */
    public static void createSubfolderWithNameOfParent(String parentFolderName, String childFolderName)
            throws Exception {
        // Родительский фид
        val parentFolderID = api(FolderList.class)
                .post().via(authClient.authHC()).getFolderId(parentFolderName);
        // Дочерняя папка
        newChildFolder(childFolderName, parentFolderID).post().via(authClient.authHC());
    }

    /**
     * Создает подпапку в папке с известным именем
     *
     * @param parentFolderFid - фид родительской папки
     * @param childFolderName - имя вложенной папки
     * @throws Exception *
     */
    public static void createSubfolderWithFidOfParent(String parentFolderFid, String childFolderName)
            throws Exception {
        // Родительское имя
        val parentFolderName = api(FolderList.class).post().via(authClient.authHC()).getFolderName(parentFolderFid);
        // Дочерняя папка
        createSubfolderWithNameOfParent(parentFolderName, childFolderName);

    }

    /**
     * Создание родительской папки с вложенной папкой
     *
     * @param parentName - имя родительской папки
     * @param childName  - имя вложенной папки
     * @throws Exception *
     */
    public static void createFolderWithNameOfParent(String parentName, String childName) throws Exception {
        // Родительский фид
        val parentFid = api(FolderList.class).post().via(authClient.authHC()).getFolderId(parentName);
        // Дочерняя папка
        newChildFolder(childName, parentFid).post().via(authClient.authHC());
    }

    /**
     * Создание вложенной структуры
     *
     * @param count     количество уровней
     * @param parentFid фид папки, где будет создана структура
     * @return String фид последней папки
     * @throws IOException *
     */
    public static String createInnerFolderStructure(String parentFid, int count) throws IOException {
        String currParentFid = parentFid;
        String currParentName = api(FolderList.class).post().via(authClient.authHC()).getFolderName(parentFid);
        String currChildName = "";

        for (int i = 0; i < count; i++) {
            currChildName = Util.getRandomString();
            newChildFolder(currChildName, currParentFid).post().via(authClient.authHC());
            currParentFid = api(FolderList.class).post().via(authClient.authHC())
                    .getFidSubfolder(currChildName, currParentName);
            currParentName = currChildName;
        }
        return currParentFid;
    }

    /**
     * Сравнение ответов 2х списков папок
     *
     * @param folderListBefore - список папок до
     * @throws Exception *
     */
    public static void currentFolderListShouldBeEqualTo(Document folderListBefore) throws Exception {
        val folderListAfter = api(FolderList.class).post().via(authClient.authHC()).toDocument();
        assertThat(folderListAfter, equalToDoc(folderListBefore).urlcomment("FolderListResp_").exclude("//scn"));
    }

    public static String takeFullFolderName(String parentName, String childName) {
        return Joiner.on("|").join(parentName, childName);
    }

    //специфичные методы для SortFolder

    /**
     * Создает в родительской папке
     * случайное количество подпапок
     * Возвращает правильный порядок папок
     *
     * @param parentFid       фид родительской папки
     * @param prevName        имя папки за которой буддем ставить папки,
     *                        нужно для формирования правильного порядка
     * @param subfolderAmount количестов создаваемых папок
     * @return возвращает отсортированный лист полных имен папок
     *         throws IOException
     */
    public static List<String> createSubfolders(String parentFid,
                                                String parentName,
                                                String prevName,
                                                int subfolderAmount) throws IOException {
        val result = new ArrayList<String>();
        String name;
        String fullName;

        for (int i = 0; i < subfolderAmount; i++) {
            name = Util.getRandomString();
            fullName = takeFullFolderName(parentName, name);
            result.add(fullName);
            newChildFolder(name, parentFid).post().via(authClient.authHC());
            assertThat(authClient.authHC(), hasSubfolder(name, parentName));
        }
        val fullPrefName = takeFullFolderName(parentName, prevName);
        result.add(fullPrefName);
        //здесь нужна сортировка, важно для логики в матчере
        Collections.sort(result);
        return result;

    }

    /**
     * Перемещает папку в текущей директории наверх,
     * заодно заполняет rightOrder.
     *
     * @param fid - фид папки
     *            throws IOException
     */
    public static List<String> move(String fid,
                                    FolderList folderList,
                                    List<String> rightOrder) throws IOException {
        firstPlaceSortOptions(fid).post().via(authClient.authHC());
        val name = folderList.getFolderName(fid);
        rightOrder.remove(name);
        rightOrder.add(0, name);
        return rightOrder;
    }

    /**
     * Перемещает папку в текущей директории в позицию,
     * перед другой папкой, заодно заполняет rightOrder.
     *
     * @param fid     - папки
     * @param prevFid throws IOException
     */
    public static List<String> move(String fid,
                                    String prevFid,
                                    FolderList folderList,
                                    List<String> rightOrder) throws IOException {
        setSortOptions(fid, prevFid).post().via(authClient.authHC());
        val name = folderList.getFolderName(fid);
        val namePrev = folderList.getFolderName(prevFid);

        rightOrder.remove(name);
        val index = rightOrder.indexOf(namePrev);
        rightOrder.add(index + 1, name);
        return rightOrder;
    }

    /**
     * Перемещает папку за папкой с неправильным фид,
     * добавляем в начало rightOrder (по договоренности с jkennedy@).
     *
     * @param fid - фид папки
     *            throws IOException
     */
    public static List<String> moveWithWrongLevel(String fid,
                                                  String prev,
                                                  FolderList folderList,
                                                  List<String> rightOrder) throws IOException {
        setSortOptions(fid, prev).post().via(authClient.authHC());
        val name = folderList.getFolderName(fid);
        rightOrder.remove(name);
        rightOrder.add(0, name);
        return rightOrder;
    }
}
