package ru.yandex.autotests.innerpochta.wmi.core.utils;

import org.apache.http.impl.client.DefaultHttpClient;
import ru.yandex.autotests.innerpochta.wmi.core.obj.MailSendMsgObj;
import ru.yandex.autotests.innerpochta.wmi.core.oper.ComposeCheck;
import ru.yandex.autotests.innerpochta.wmi.core.oper.MailBoxList;
import ru.yandex.autotests.innerpochta.wmi.core.oper.MailSend;
import ru.yandex.autotests.innerpochta.wmi.core.oper.FolderList;
import ru.yandex.autotests.innerpochta.wmi.core.oper.ThreadsView;
import ru.yandex.autotests.innerpochta.wmi.core.response.MailboxListResponseItem;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;
import ru.yandex.autotests.innerpochta.wmi.core.rules.InitialComposeCheck;
import ru.yandex.autotests.innerpochta.wmi.core.rules.InitialFolderList;
import ru.yandex.qatools.allure.annotations.Step;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.api;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.jsx;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 9/13/13
 * Time: 5:15 PM
 */
public class SendUtils {
    //инициализируем в конструкторе
    private static HttpClientManagerRule authClient;
    private MailSendMsgObj msg;

    private ComposeCheck composeCheck;
    private InitialComposeCheck initialComposeCheck;

    private FolderList folderList;
    private InitialFolderList initialFolderList;
    //храним тему для получения по ней mid-ов и tid-ов
    private String lastSubject;

    //апи метод для того чтобы можно было отправлять через продакшн
    private MailSend sendOper;

    private AbstractWaitSender wait;
    private int count;

    public SendUtils(HttpClientManagerRule authClient) {
        this.authClient = authClient;
        //заполняем исходные данные
        reset();
    }

    public SendUtils folderList(FolderList folderList) {
        this.folderList = folderList;
        return this;
    }

    public SendUtils folderList(InitialFolderList folderList) {
        this.initialFolderList = folderList;
        return this;
    }

    public SendUtils composeCheck(ComposeCheck composeCheck) {
        this.composeCheck = composeCheck;
        return this;
    }

    public SendUtils composeCheck(InitialComposeCheck composeCheck) {
        this.initialComposeCheck = composeCheck;
        return this;
    }

    private abstract class AbstractWaitSender {
        abstract void waitMails(String subject, int count, DefaultHttpClient hc) throws IOException;
    }

    private class NoWaitSender extends AbstractWaitSender {
        void waitMails(String subject, int count, DefaultHttpClient hc) {
            //DO NOTHING
        }
    }

    private class WaitSender extends AbstractWaitSender {
        void waitMails(String subject, int count, DefaultHttpClient hc) throws IOException {
            new WaitUtils(authClient).folderList(initialFolderList).folderList(folderList)
                    .subj(subject).count(count).usingHC(hc).waitDeliver();
        }
    }

    /**
     * @param subj - тема письма
     * @return this
     */
    public SendUtils subj(String subj) {
        if (null != subj) {
            this.lastSubject = subj;
            msg.setSubj(subj);
        }
        return this;
    }

    /**
     * @param to адрес получателя. Можно указывать несколько через запятую
     * @return this
     */
    public SendUtils to(String to) {
        if (null != to) {
            msg.setTo(to);
        }
        return this;
    }

    public SendUtils cc(String cc) {
        if (null != cc) {
            msg.setCc(cc);
        }
        return this;
    }

    public SendUtils bcc(String bcc) {
        if (null != bcc) {
            msg.setCc(bcc);
        }
        return this;
    }

    /**
     * @param send - тело письма
     * @return
     */
    public SendUtils send(String send) {
        if (null != send) {
            msg.setSend(send);
        }
        return this;
    }

    public SendUtils tTypeHtml() {
        msg.setTtypeHtml();
        return this;
    }

    public SendUtils nosend(Boolean yes) {
        if (yes) {
            msg.setNosend("yes");
        } else {
            msg.setNosend("no");
        }
        return this;
    }

    public SendUtils nosave(Boolean yes) {
        if (yes) {
            msg.setNosave("yes");
        } else {
            msg.setNosave("no");
        }
        return this;
    }

    public SendUtils ignOverwrite(Boolean yes) {
        if (yes) {
            msg.setIgnOverwrite("yes");
        } else {
            msg.setIgnOverwrite("no");
        }
        return this;
    }

    public SendUtils addAtts(File... file) throws FileNotFoundException {
        if (null != file) {
            msg.addAtts(file);
        }
        return this;
    }

    public SendUtils addRotate(String rotate) throws FileNotFoundException {
        if (null != rotate) {
            msg.addRotate(rotate);
        }
        return this;
    }

    public SendUtils addAtts(String mimetype, File... file) throws FileNotFoundException {
        if (null != file) {
            msg.addAtts(mimetype, file);
        }
        return this;
    }

    public SendUtils addParts(String parts) {
        if (null != parts) {
            msg.addParts(parts);
        }
        return this;
    }

    public SendUtils addIds(String... ids) {
        if (null != ids) {
            msg.addIds(ids);
        }
        return this;
    }

    public SendUtils inreplyto(String inreplyto) {
        if (null != inreplyto) {
            msg.setInreplyto(inreplyto);
        }
        return this;
    }

    public SendUtils overwrite(String mid) {
        if (null != mid) {
            msg.setOverwrite(mid);
        }
        return this;
    }

    //replied/forwarded
    public SendUtils markAs(String mark) {
        if (null != mark) {
            msg.setMarkAs(mark);
        }
        return this;
    }

    public SendUtils markIds(String mid) {
        if (null != mid) {
            msg.setMarkIds(mid);
        }
        return this;
    }

    /**
     * @param msg - объект MailSendMsgObj можно сразу передать
     * @return
     */
    public SendUtils msg(MailSendMsgObj msg) {
        this.msg = msg;
        return this;
    }

    /**
     * @param count - устанавливаем количество писем
     * @return this
     */
    public SendUtils count(int count) {
        this.count = count;
        return this;
    }

    /**
     * Ждем пока письмо придет во входящие
     * Для писем отправленных самому себе
     *
     * @return this
     */
    public SendUtils waitDeliver() {
        wait = new WaitSender();
        return this;
    }

    /**
     * Устанавливаем настройку для отправления
     * письма через продакшн
     *
     * @return this
     */
    public SendUtils viaProd() {
        sendOper.setHost(props().productionHost());
        return this;
    }

    /**
     * Синтаксический сахар
     *
     * @return
     */
    public SendUtils and() {
        return this;
    }

    /**
     * Добавляем текст письма
     *
     * @return this
     */
    public SendUtils setSend(String text) {
        this.msg.setSend(text);
        return this;
    }

    /**
     * Сбрасываем настройки рулы, так как в руле хранится информация об отправленных письмах,
     * то следует сбрасывать после каждого теста
     *
     * @return this
     */
    private SendUtils reset() {
        this.msg = MailSendMsgObj.msg().setTo(authClient.acc().getSelfEmail());
        this.wait = new NoWaitSender();
        this.count = 1;
        //сбрасываем настройки операции, чтобы не отправлять через прод
        this.sendOper = api(MailSend.class);
        return this;
    }

    @Step("[SEND]: Отправляем письма ({0}) с темой ({1}) пользователю ({2})")
    private SendUtils send(int count, String subj, String to) throws Exception {

        if (null != initialComposeCheck) {
            composeCheck = initialComposeCheck.get();
        }

        if (null == composeCheck) {
            composeCheck = api(ComposeCheck.class).post().via(authClient.authHC());
        }

        sendOper.params(msg.setComposeCheck(composeCheck.getComposeCheckNodeValue()));
        for (int i = 0; i < count; ++i) {
            sendOper.post().via(authClient.authHC()).resultOk();
        }

        wait.waitMails(subj, count, authClient.authHC());
        return this;
    }

    /**
     * Отправляем сформированное/ые письмо/а
     *
     * @return this
     * @throws Exception
     */
    public SendUtils send() throws Exception {
        send(count, msg.getSubj(), msg.getTo());
        //после каждой отправки сбрасываем настройки рулы и заполняем поля
        this.lastSubject = msg.getSubj();
        reset();
        return this;
    }

    public String getSubj() {
        return lastSubject;
    }

    /**
     * Возвращаем миды писем
     *
     * @return
     * @throws Exception
     */
    public List<String> getMids() throws Exception {
        return jsx(MailBoxList.class).post().via(authClient.authHC())
                .getMidsOfMessagesWithSubject(lastSubject);
    }

    /**
     * Возвращаем первый найденный мид
     *
     * @return
     * @throws Exception
     */
    public String getMid() throws IOException {
        return jsx(MailBoxList.class).post().via(authClient.authHC()).getMidOfMessage(lastSubject);
    }

    /**
     * Возвращаем первый найденный stid
     *
     * @return stid последнего отправленного письма
     * @throws Exception
     */
    public String getStid() throws IOException {
        return jsx(MailBoxList.class).post().via(authClient.authHC()).getStidOfMessage(lastSubject);
    }

    /**
     * Возвращаем received_date последнего отправленного письма
     */
    public Date getRecvDate() throws IOException, ParseException {
        final String str = jsx(MailBoxList.class).post().via(authClient.authHC()).getRecvDateBySubject(lastSubject);
        DateFormat format  = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
        return format.parse(str);
    }

    /**
     * Возвращаем tid письма, группы писем
     *
     * @return
     * @throws Exception
     */
    public String getTid() throws IOException {
        return jsx(ThreadsView.class).post().via(authClient.authHC()).getThreadId(lastSubject);
    }

    public MailboxListResponseItem getMessage() {
        return jsx(MailBoxList.class).post().via(authClient.authHC()).getMessageBySubject(lastSubject);
    }
}
