package ru.yandex.autotests.innerpochta.wmi.core.utils;

import org.apache.commons.io.FileUtils;

import ru.yandex.autotests.innerpochta.beans.sendbernar.SendMessageResponse;
import ru.yandex.autotests.innerpochta.beans.sendbernar.WriteAttachmentResponse;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;
import ru.yandex.autotests.innerpochta.wmi.core.sendbernar.DiskAttachHelper;
import ru.yandex.autotests.innerpochta.wmi.core.sendbernar.savedraft.ApiSaveDraft;
import ru.yandex.autotests.innerpochta.wmi.core.sendbernar.sendmessage.ApiSendMessage;
import ru.yandex.autotests.innerpochta.wmicommon.Util;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.google.common.base.Joiner.on;
import static ru.yandex.autotests.innerpochta.wmi.core.api.CommonApiSettings.shouldBe;
import static ru.yandex.autotests.innerpochta.wmi.core.api.WmiApis.apiSendbernar;
import static ru.yandex.autotests.innerpochta.wmi.core.base.props.WmiCoreProperties.props;
import static ru.yandex.autotests.innerpochta.wmi.core.sendbernar.SendbernarResponses.ok200;

public class SendbernarUtils {
    private SendbernarUtils(HttpClientManagerRule authClient) {
        this.authClient = authClient;
        wait = new WaitForMessage(authClient);
        msg = new Message(authClient);
        count = 1;
        sendbernarHost = props().sendbernarUri();
    }

    public static SendbernarUtils sendWith(HttpClientManagerRule authClient) {
        return new SendbernarUtils(authClient);
    }

    public SendbernarUtils viaProd() {
        return this;
    }

    public SendbernarUtils count(int count) {
        this.count = count;
        return this;
    }

    public SendbernarUtils subj(String subj) {
        if (null != subj) {
            msg.subject = subj;
        }
        return this;
    }

    public SendbernarUtils addAttaches(List<File> files) {
        if (null != files) {
            msg.attaches.addAll(files);
        }
        return this;
    }

    public SendbernarUtils addAttaches(File... file) {
        if (null != file) {
            msg.attaches.addAll(Arrays.asList(file));
        }
        return this;
    }

    public SendbernarUtils addDiskAttaches(DiskAttachHelper... attach) {
        if (null != attach) {
            msg.diskAttaches.addAll(Arrays.asList(attach).stream().map(DiskAttachHelper::getJson).collect(Collectors.toList()));
        }
        return this;
    }

    public SendbernarUtils addFwdMids(String... mids) {
        if (null != mids) {
            msg.fwdMids.addAll(Arrays.asList(mids));
        }
        return this;
    }

    public SendbernarUtils inReplyTo(String mid) {
        msg.inReplyToMid = Optional.ofNullable(mid);
        return this;
    }

    public SendbernarUtils to(String to) {
        if (null != to) {
            msg.to = to;
        }
        return this;
    }

    public SendbernarUtils cc(String cc) {
        msg.cc = Optional.ofNullable(cc);
        return this;
    }

    public SendbernarUtils bcc(String bcc) {
        msg.bcc = Optional.ofNullable(bcc);
        return this;
    }

    public SendbernarUtils text(String text) {
        if (null != text) {
            msg.text = text;
        }
        return this;
    }

    public SendbernarUtils html(String html) {
        msg.html = Optional.ofNullable(html);
        return this;
    }

    public SendbernarUtils fromName(String fromName) {
        msg.fromName = Optional.ofNullable(fromName);
        return this;
    }

    public SendbernarUtils mentions(String ment) {
        msg.mentions = Optional.ofNullable(ment);
        return this;
    }

    public WaitForMessage saveDraft() {
        wait.subj(msg.subject).count(count).draft();
        for (int i = 0; i < count; ++i) {
            ApiSaveDraft apiSaveDraft = apiSendbernar(authClient.account().userTicket(), sendbernarHost).saveDraft()
                    .withDefaults()
                    .withUid(uid())
                    .withText(msg.text)
                    .withSubj(msg.subject)
                    .withTo(msg.to);
            msg.attaches.stream()
                    .map(this::uploadedId)
                    .forEach(apiSaveDraft::withUploadedAttachStids);
            msg.fwdMids.forEach(apiSaveDraft::withForwardMids);
            msg.inReplyToMid.map(apiSaveDraft::withInreplyto);
            msg.html.map(apiSaveDraft::withHtml);
            msg.fromName.map(apiSaveDraft::withFromName);
            msg.cc.map(apiSaveDraft::withCc);
            msg.bcc.map(apiSaveDraft::withBcc);
            apiSaveDraft.post(shouldBe(ok200()));
        }
        return wait;
    }

    public WaitForMessage send() {
        wait.subj(msg.subject).count(count).inbox();
        List<String> references = new ArrayList<>();
        for (int i = 0; i < count; ++i) {
            ApiSendMessage apiSendMessage = apiSendbernar(authClient.account().userTicket(), sendbernarHost).sendMessage()
                    .withDefaults()
                    .withUid(uid())
                    .withText(msg.text)
                    .withSubj(msg.subject)
                    .withTo(msg.to);
            if (msg.mentions.isPresent()) {
                apiSendMessage = apiSendMessage.withMentions(msg.mentions.get());
            }
            msg.attaches.stream()
                    .map(this::uploadedId)
                    .forEach(apiSendMessage::withUploadedAttachStids);
            if (!msg.diskAttaches.isEmpty()) {
                msg.diskAttaches.forEach(apiSendMessage::withDiskAttachesJson);
            }
            msg.fwdMids.forEach(apiSendMessage::withForwardMids);
            if (msg.inReplyToMid.isPresent()) {
                apiSendMessage.withInreplyto(msg.inReplyToMid.get());
                apiSendMessage.withMarkAs("replied");
                apiSendMessage.withReferences(msg.inReplyToMid.get());
            } else if (!references.isEmpty()) {
                apiSendMessage.withInreplyto(references.get(references.size() - 1));
                apiSendMessage.withMarkAs("replied");
                apiSendMessage.withReferences(on(" ").join(references));
            }
            msg.html.map(apiSendMessage::withHtml);
            msg.fromName.map(apiSendMessage::withFromName);
            msg.cc.map(apiSendMessage::withCc);
            msg.bcc.map(apiSendMessage::withBcc);
            references.add(apiSendMessage.post(shouldBe(ok200()))
                    .as(SendMessageResponse.class)
                    .getMessageId());

        }
        return wait;
    }

    public String writeAttachment(File attach) {
        return uploadedId(attach);
    }

    public String getSubj() {
        return msg.subject;
    }

    private String uid() {
        return authClient.account().uid();
    }

    private String uploadedId(File attach) {
        byte[] content;
        try {
            content = FileUtils.readFileToByteArray(attach);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return apiSendbernar(authClient.account().userTicket(), sendbernarHost).writeAttachment()
                .withUid(uid())
                .withFilename(attach.getName())
                .withReq((req) -> req.setBody(content))
                .post(shouldBe(ok200()))
                .as(WriteAttachmentResponse.class)
                .getId();
    }

    private class Message {
        Message(HttpClientManagerRule authClient) {
            to = authClient.acc().getSelfEmail();
            cc = Optional.empty();
            bcc = Optional.empty();
            subject = Util.getRandomString();
            attaches = new ArrayList<>();
            diskAttaches = new ArrayList<>();
            text = "";
            html = Optional.empty();
            fwdMids = new ArrayList<>();
            inReplyToMid = Optional.empty();
            mentions = Optional.empty();
            fromName = Optional.empty();
        }

        String subject;
        String to;
        Optional<String> cc;
        Optional<String> bcc;
        List<File> attaches;
        List<String> diskAttaches;
        String text;
        Optional<String> html;
        List<String> fwdMids;
        Optional<String> inReplyToMid;
        Optional<String> mentions;
        Optional<String> fromName;
    }

    private HttpClientManagerRule authClient;
    private WaitForMessage wait;
    private int count;
    private String sendbernarHost;
    private Message msg;
}
