package ru.yandex.autotests.innerpochta.wmi.core.utils;

import ru.yandex.autotests.innerpochta.beans.folderlist.Symbol;
import ru.yandex.autotests.innerpochta.wmi.core.exceptions.RetryException;
import ru.yandex.autotests.innerpochta.wmi.core.matchers.messages.IsThereMessagesMatcher;
import ru.yandex.autotests.innerpochta.wmi.core.obj.hound.FoldersObj;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;
import ru.yandex.autotests.innerpochta.wmicommon.WmiConsts;
import ru.yandex.qatools.allure.annotations.Step;

import static java.util.concurrent.TimeUnit.SECONDS;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.hamcrest.MatcherAssert.assertThat;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.WaitForMatcherDecorator.withWaitFor;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.messages.IsThereMessagesMatcher.hasMsgsIn;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.messages.IsThereMessagesMatcher.hasMsgsStrictSubjectIn;
import static ru.yandex.autotests.innerpochta.wmi.core.oper.hound.Folders.folders;

public class WaitForMessage {
    private HttpClientManagerRule authClient;
    private String subj;
    private boolean strictSubjectChecking = false;
    private String errorMsg;
    private long timeout;
    private int count;
    private String fid;

    private void reset() {
        this.subj = WmiConsts.NO_SUBJECT_TITLE;
        this.errorMsg = "";
        this.timeout = SECONDS.toMillis(60);
        this.count = 1;
        this.fid = null;
    }

    private WaitForMessage(WaitForMessage other, HttpClientManagerRule authClient) {
        this.authClient = authClient;
        this.subj = other.subj;
        this.errorMsg = other.errorMsg;
        this.timeout = other.timeout;
        this.count = other.count;
        this.fid = other.fid;
    }

    private String fid(Symbol sybmol) {
        return folders(FoldersObj.empty().setUid(authClient.account().uid()))
                .get()
                .via(authClient)
                .fid(sybmol);
    }

    public WaitForMessage(HttpClientManagerRule authClient) {
        this.authClient = authClient;
        reset();
    }

    public WaitForMessage subj(String subj) {
        this.subj = subj;
        return this;
    }

    public WaitForMessage strict() {
        strictSubjectChecking = true;
        return this;
    }

    public WaitForMessage inbox() {
        this.fid = fid(Symbol.INBOX);
        return this;
    }

    public WaitForMessage sent() {
        this.fid = fid(Symbol.SENT);
        return this;
    }

    public WaitForMessage spam() {
        this.fid = fid(Symbol.SPAM);
        return this;
    }

    public WaitForMessage draft() {
        this.fid = fid(Symbol.DRAFT);
        return this;
    }

    public WaitForMessage fid(String fid) {
        this.fid = fid;
        return this;
    }

    public WaitForMessage template() {
        this.fid = fid(Symbol.TEMPLATE);
        return this;
    }

    public WaitForMessage usingHttpClient(HttpClientManagerRule client) {
        return new WaitForMessage(this, client);
    }

    public WaitForMessage errorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
        return this;
    }

    public WaitForMessage timeout(long timeout) {
        this.timeout = timeout;
        return this;
    }

    public WaitForMessage count(int count) {
        this.count = count;
        return this;
    }

    public WaitedMailbox waitDeliver() {
        if (fid == null) {
            inbox();
        }

        if (null == subj) {
            waitMessagesDeliver(count, fid, timeout, errorMsg);
        } else {
            waitMessagesDeliver(subj, count, fid, timeout, errorMsg);
        }

        WaitedMailbox mailbox = new WaitedMailbox(subj, fid, authClient);
        reset();

        return mailbox;
    }

    @Step("[WAIT]: Ждем письма ({1}) с темой \"{0}\" в папке \"{3}\" <{2}>")
    private void waitMessagesDeliver(String subj, Integer count, String fid,
                                     long timeout, String failMsg) throws RetryException {
        try {
            String msg = isEmpty(failMsg)
                    ? "Не все письма с темой '" + subj + "' были найдены в папке " + fid + "  ."
                    : failMsg;

            IsThereMessagesMatcher matcher = strictSubjectChecking
                    ? hasMsgsStrictSubjectIn(subj, count, fid)
                    : hasMsgsIn(subj, count, fid);
            assertThat(msg,
                    authClient,
                    withWaitFor(matcher, timeout));
        } catch (AssertionError error) {
            throw new RetryException(error);
        }
    }

    @Step("[WAIT]: Ждем письма ({0}) в папке \"{2}\" <{3}>")
    private void waitMessagesDeliver(Integer count, String fid,
                                     long timeout, String failMsg) throws RetryException {
        try {
            String msg = isEmpty(failMsg)
                    ? "Количество писем в папке '" + fid + "' не совпадает с ожидаемым."
                    : failMsg;

            assertThat(msg,
                    authClient,
                    withWaitFor(hasMsgsIn(count, fid), timeout));
        } catch (AssertionError error) {
            throw new RetryException(error);
        }
    }
}
