package ru.yandex.autotests.innerpochta.wmi.core.utils;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import ru.yandex.autotests.innerpochta.wmi.core.exceptions.RetryException;
import ru.yandex.autotests.innerpochta.wmi.core.obj.MailBoxListObj;
import ru.yandex.autotests.innerpochta.wmi.core.oper.FolderList;
import ru.yandex.autotests.innerpochta.wmi.core.oper.MailBoxList;
import ru.yandex.autotests.innerpochta.wmi.core.oper.ThreadsView;
import ru.yandex.autotests.innerpochta.wmi.core.response.MailboxListResponseItem;
import ru.yandex.autotests.innerpochta.wmi.core.rules.InitialFolderList;
import ru.yandex.autotests.innerpochta.wmi.core.rules.HttpClientManagerRule;
import ru.yandex.qatools.allure.annotations.Step;

import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

import static java.util.concurrent.TimeUnit.SECONDS;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.hamcrest.MatcherAssert.assertThat;
import static ru.yandex.autotests.innerpochta.wmi.core.base.Exec.jsx;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.WaitForMatcherDecorator.withWaitFor;
import static ru.yandex.autotests.innerpochta.wmi.core.matchers.messages.IsThereMsgsMatcher.hasMsgsIn;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 09.10.14
 * Time: 17:26
 */
public class WaitUtils {

    protected static final Logger logger = LogManager.getLogger(WaitUtils.class);

    //по умолчанию если темы нет, выставляется такой subject
    private String subj = "No subject";
    private int count = 1;
    private String fid;
    private String failMsg;

    private FolderList folderList;
    private InitialFolderList initialFolderList;

    private HttpClientManagerRule authClient;
    private DefaultHttpClient hc;

    private DefaultHttpClient lastHc;
    private String lastFid;
    private String lastSubject;
    private long timeout = SECONDS.toMillis(60);



    public WaitUtils() {
    }

    public WaitUtils(HttpClientManagerRule authClient) {
        this.authClient = authClient;
    }

    public static WaitUtils waitFor() {
        return new WaitUtils();
    }

    public WaitUtils folderList(FolderList folderList) {
        if (null != folderList) {
            this.folderList = folderList;
        }
        return this;
    }

    public WaitUtils folderList(InitialFolderList folderList) {
        if (folderList != null) {
            this.initialFolderList = folderList;
        }
        return this;
    }

    public WaitUtils errorMsg(String errorMsg) {
        if (null != errorMsg) {
            this.failMsg = errorMsg;
        }
        return this;
    }

    public WaitUtils subj(String subj) {
        if (null != subj) {
            this.subj = subj;
        }
        return this;
    }

    public WaitUtils timeout(long timeout) {
        this.timeout = timeout;
        return this;
    }

    public WaitUtils count(Integer count) {
        if (null != count) {
            this.count = count;
        }
        return this;
    }

    public WaitUtils inFid(String fid) {
        if (null != fid) {
            this.fid = fid;
        }
        return this;
    }

    public WaitUtils usingHC(DefaultHttpClient hc) {
        if (hc != null) {
            this.hc = hc;
            folderList = null;
            initialFolderList = null;
        }
        return this;
    }

    /**
     * Сбрасываем настройки рулы
     *
     * @return this
     */
    public WaitUtils reset() {
        this.subj = "No subject";
        this.hc = null;
        this.fid = null;
        this.count = 1;
        return this;
    }

    /**
     * Синтаксический сахар
     *
     * @return
     */
    public WaitUtils and() {
        return this;
    }


    @Step("[WAIT]: Ждем письма ({1}) с темой \"{0}\" в папке \"{3}\" <{2}>")
    private void waitMsgDeliver(String subj, Integer count, String fid, long timeout, String folderName,
                                String failMsg, DefaultHttpClient hc) {
        waitMsgDeliver(subj, count, fid, timeout, failMsg, hc);
    }


    /**
     * Ожидание доставки письма в любой папке c указанным сообщением об ошибке
     * использует указанный клиент
     *
     * @param subj    - тема письма
     * @param count   - количество писем с данной темой
     * @param fid     - Фид папки, где ищем
     * @param failMsg - сообщение в случае, если не дождемся
     * @throws ru.yandex.autotests.innerpochta.wmi.core.exceptions.RetryException
     */
    private void waitMsgDeliver(String subj, Integer count, String fid, long timeout,
                                String failMsg, DefaultHttpClient hc)
            throws RetryException {

        try {
            assertThat((isEmpty(failMsg))
                            ? "Не все письма с темой '" + subj + "' были найдены."
                            : failMsg, hc,
                    withWaitFor(hasMsgsIn(subj, count, fid), timeout));
        } catch (AssertionError error) {
            //если не дождались, сбрасываем настройки рулы
            reset();
            throw new RetryException(error);
        }
    }

    public WaitUtils waitDeliver() throws IOException {
        if (null == hc) {
            hc = authClient.authHC();
        }

        if (null != initialFolderList) {
            folderList = initialFolderList.get();
        }

        if (null == folderList) {
            folderList = jsx(FolderList.class).post().via(hc);
        }

        //по умолчанию берем папку "Входящие"
        if (null == fid) {
            fid = folderList.defaultFID();
        }

        waitMsgDeliver(subj, count, fid, timeout, StringUtils.defaultIfEmpty(folderList.getFolderName(fid), "..."), failMsg, hc);
        //сбрасываем все настройки после каждого успешного ожидания

        this.lastHc = hc;
        this.lastSubject = subj;
        this.lastFid = fid;
        reset();
        return this;
    }

    /**
     * Возвращаем миды писем
     *
     * @return
     * @throws Exception
     */
    public List<String> getMids() throws Exception {
        return jsx(MailBoxList.class).params(MailBoxListObj.empty().setCurrentFolder(lastFid)).post().via(lastHc)
                .getMidsOfMessagesWithSubject(lastSubject);
    }

    /**
     * Возвращаем первый найденный мид
     *
     * @return
     * @throws Exception
     */
    public String getMid() throws IOException {
        return jsx(MailBoxList.class).params(MailBoxListObj.empty().setCurrentFolder(lastFid)).post().via(lastHc)
                .getMidOfMessage(lastSubject);
    }

    /**
     * Возвращаем tid письма, группы писем
     *
     * @return
     * @throws Exception
     */
    public String getTid() throws IOException {
        return jsx(ThreadsView.class).params(MailBoxListObj.empty().setCurrentFolder(lastFid)).post().via(lastHc)
                .getThreadId(lastSubject);
    }

    /**
     * Возвращаем первый найденный стид
     */
    public String getStid() throws IOException {
        return jsx(MailBoxList.class).params(MailBoxListObj.empty().setCurrentFolder(lastFid)).post().via(lastHc)
                .getStidOfMessage(lastSubject);
    }

    /**
     * Возвращаем received_date последнего доставленного письма
     */
    public Date getRecvDate() throws IOException, ParseException {
        final String str = jsx(MailBoxList.class)
                .params(MailBoxListObj.empty().setCurrentFolder(lastFid))
                .post().via(lastHc)
                .getRecvDateBySubject(lastSubject);
        DateFormat format  = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
        return format.parse(str);
    }

    @Step("Ждем {0} {1} ...")
    public static void waitSmthStep(long time, String descr, TimeUnit unit) {
        //делаем таймаут
        try {
            logger.info(String.format("WAIT %d %s...", time, unit.toString()));
            Thread.sleep(unit.toMillis(time));
        } catch (InterruptedException e) {

        }
    }


    public static void waitSmth(long time, TimeUnit unit) {
        waitSmthStep(time, unit.name(), unit);
    }

    public MailboxListResponseItem getMessage() {
        return jsx(MailBoxList.class)
                .params(MailBoxListObj.empty().setCurrentFolder(lastFid))
                .post().via(lastHc)
                .getMessageBySubject(lastSubject);
    }
}
