// **************
// HELPFUL LINKS
// **************
// https://jenkinsci.github.io/job-dsl-plugin
// https://github.yandex-team.ru/qatools/startrek-jenkins-plugin/blob/fa16a72deed91b3cab7493717a87156f63076202/README.md

def create_job = { name, envs, cloud = 'qloud' ->
freeStyleJob(name) {
  description("""
Сборка докер-образов для сборки проектов
<br/>
Эта джоба сгенерирована автоматически. <br/>
Подробности смотреть в <a href=${JOB_URL}>Seed Job</a>
<br/>""")

  label('xiva_deb')
  logRotator(14, 200)
  concurrentBuild(false)
  parameters {
      stringParam('BRANCH', 'master', 'ветка в репозитории, которую собирать')
      booleanParam('CLEAN_WORKING_DIR', false, 'снести сборочную директорию')
  }
  environmentVariables(envs)

  wrappers {
      colorizeOutput('xterm')
      sshAgent('robot-gerrit-ssh-key')

      credentialsBinding {
          usernamePassword('DOCKER_REGISTRY_USER', 'DOCKER_REGISTRY_TOKEN', 'robot-sender-docker-registry-creds')
      }

      preBuildCleanup {
          deleteDirectories()
          cleanupParameter('CLEAN_WORKING_DIR')
      }
  }

  scm {
      git {
          remote {
              url('git@bb.yandex-team.ru:mail/arcadia.git')
              credentials('robot-gerrit-ssh-key')
          }
          branch('${BRANCH}')
          extensions {
              // dot not shallow clone because shallow updates are not allowed in git repos
              submoduleOptions {
                  recursive()
              }
          }
      }
  }
  steps {
      shell('''
        set -eo pipefail
        DT=`date "+%Y-%m-%d-%H-%M"`;
        VERSION="${DT}${BRANCH}"
        BUILDNAME="${name}:r${VERSION}"
        echo "VERSION=${VERSION}" > variables.txt
        echo "GITTAG=${name}-${VERSION}" >> variables.txt
        echo "BUILDNAME=${BUILDNAME}" >> variables.txt
        echo "BUILDTAG=registry.yandex.net/mail/xiva/${BUILDNAME}" >> variables.txt
        echo "BUILDARGS=--checkout -j`nproc --ignore=10` --build=relwithdebinfo" >> variables.txt
      ''')

      environmentVariables {
          propertiesFile('variables.txt')
      }

      shell('''
        set -eo pipefail
        [ -d ~/asg ] && ~/asg/asg selfupdate
        [ -d ~/asg ] || git clone --depth 1 https://github.yandex-team.ru/xiva/asg ~/asg && ~/asg/asg-integrate-bash
      ''')

      shell('''
        set -eo pipefail
        [ -d build ] || ~/asg/asg clone --branch ${BRANCH} build
        cd build
        cd mail
        git fetch
        git reset --hard
        git clean -dfx
        git checkout origin/${BRANCH}
        cd ..
        ~/asg/asg up
      ''')

      shell('''
        set -eo pipefail
        [ "${SKIP_BUILD}" == "1" ] && { echo "skip build"; exit 0; }
        cd build
        ~/asg/asg make "${BUILDARGS}" "${project_directory}" mail/yplatform/config-dumper/ mail/pa/
      ''')

      shell('''
        set -eo pipefail
        [ "${SKIP_UNIT_TESTS}" == "1" ] && { echo "skip tests"; exit 0; }
        cd build
        ~/asg/asg make "${BUILDARGS}" -t "${project_directory}"
      ''')

      shell('''
        set -eo pipefail
        [ "${SKIP_SYSTEM_TESTS}" == "1" ] && { echo "skip system tests"; exit 0; }
        cd build
        cd "${project_directory}/tests-system"
        if [ -f "./run.sh" ]; then
            BUILD_ROOT_DIR="../" sh ./run.sh || BUILD_ROOT_DIR="../" sh ./run.sh
        else
            ~/asg/asg make "${BUILDARGS}" -t --test-size=LARGE
        fi
      ''')

      def prepare_destdir_steps = '''
        copy() {
            src="$1"
            dst="$2"
            mkdir -p "$dst"
            cp -r "$src" "$dst"
        }

        cd build
        rm -rf destdir
        cp -r "${docker_common_directory}/deploy" "${project_directory}"
        mkdir destdir
        mkdir -p destdir/usr/bin/
        mkdir -p destdir/usr/sbin/
        cp ya destdir/usr/bin/
        for i in ${paths_to_copy[@]}; do copy $(echo $i | tr "," " "); done
      '''

      def copy_common_binaries_steps = '''
        cp mail/yplatform/config-dumper/config-dumper destdir/usr/bin/
        cp mail/pa/pa/pa destdir/usr/bin/
      '''

      if (cloud == 'qloud') {
        shell("""
            set -eo pipefail
            docker_common_directory=mail/rtec/docker-common
            ${prepare_destdir_steps}
            cp -Lr "\${project_directory}/etc" destdir/

            [ "\${SKIP_BUILD}" == "1" ] && { echo "build skipped, no binaries copied"; exit 0; }
            cp "\${project_directory}/\${project_binary}" destdir/usr/sbin/
            ${copy_common_binaries_steps}
        """)
      } else if (cloud == 'deploy') {
          shell("""
            set -eo pipefail
            docker_common_directory=mail/rtec/docker-common-deploy
            ${prepare_destdir_steps}
            mkdir -p destdir/app/
            cp -Lr "\${project_directory}/config" destdir/app/config/

            [ "\${SKIP_BUILD}" == "1" ] && { echo "build skipped, no binaries copied"; exit 0; }
            cp "\${project_directory}/\${project_binary}" destdir/app/
            ${copy_common_binaries_steps}
        """)
      } else {
          throw new Exception("Unknown cloud: ${cloud}")
      }

      shell('''
        set -eo pipefail
        pushd build
        docker build --pull --network=host --build-arg installdir="destdir" --build-arg deploydir="${project_directory}/deploy" --tag ${BUILDTAG} -f ${project_directory}/Dockerfile .
        docker push ${BUILDTAG}
        popd
        DIGEST=$(docker inspect -f '{{.RepoDigests}}' ${BUILDTAG} | awk -F '(\\\\[|@|\\\\])' '{print $3}')
        echo "DIGEST=$DIGEST" >> variables.txt
      ''')

      environmentVariables {
          propertiesFile('variables.txt')
      }

      publishers {
          postBuildTask {
              task('.*', '''
                  # cleanup docker containers and images after build
                  docker stop $(docker ps -a -q) || true # do not fail if no running containers left
                  docker rm -f $(docker ps -a -q) || true
                  docker container prune --force
                  docker image rm ${BUILDTAG} || true # do not fail if image was not build successfully
                  docker image prune --force
                  docker volume prune --force
              ''')
          }
      }

      publishers {
          git {
              pushOnlyIfSuccess()
              tag('origin', '${GITTAG}') {
                  create(true)
                  update(true)
              }
          }
      }

      publishers {
          buildDescription('', '''${BUILDTAG} ${DIGEST}''')
      }
  }

  wrappers {
    buildName('${BUILDNAME}')
  }

}
}

create_job('b_xiva_reaper', [
  'name':'reaper',
  'project_directory':'mail/xiva/reaper',
  'project_binary': 'reaper'
])

create_job('b_xiva_mesh', [
  'name':'xiva-mesh',
  'project_directory':'mail/xiva/mesh',
  'project_binary': 'xivamesh'
])

create_job('b_xiva_mob', [
  'name':'xiva-mobile',
  'project_directory':'mail/xiva/mobile',
  'project_binary': 'app'
], 'deploy')

create_job('b_xiva_conf', [
  'name':'xiva-conf',
  'project_directory':'mail/xiva/conf',
  'project_binary': 'xivaconf',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_xiva_server', [
  'name':'xiva-server',
  'project_directory':'mail/xiva/server',
  'project_binary': 'xiva-server'
])

create_job('b_xiva_hub', [
  'name':'xiva-hub',
  'project_directory':'mail/xiva/hub',
  'project_binary': 'xivahub'
])

create_job('b_xiva_equalizer', [
  'name':'xiva-equalizer',
  'project_directory':'mail/xiva/equalizer',
  'project_binary': 'equalizer',
  'paths_to_copy': 'mail/rtec/tools/shards_analyzer.py,${project_directory}/deploy/usr/share/equalizer/',
  'SKIP_UNIT_TESTS': '1',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_xeno', [
  'name':'xeno',
  'project_directory':'mail/xeno',
  'project_binary': 'xeno',
  'paths_to_copy': 'mail/macs_pg/etc/query.conf,${project_directory}/etc/${name}/ mail/rtec/tools/shards_analyzer.py,${project_directory}/deploy/usr/share/${name}/ mail/ymod_cachedb/etc/cachedb.conf,${project_directory}/etc/${name}/'
])

create_job('b_rcache', [
  'name':'rcache',
  'project_directory':'mail/replicated_cache',
  'project_binary': 'rcache'
])

create_job('b_imap', [
  'name':'imap',
  'project_directory':'mail/imap',
  'project_binary': 'imap',
  'paths_to_copy': 'mail/macs_pg/etc/query.conf,${project_directory}/etc/imap/'
])

create_job('b_pop3', [
  'name':'pop3',
  'project_directory':'mail/pop3',
  'project_binary': 'app',
  'SKIP_SYSTEM_TESTS': '1'
], 'deploy')

create_job('b_collectors', [
  'name':'collectors',
  'project_directory':'mail/collectors',
  'project_binary': 'collectors',
  'paths_to_copy': 'mail/macs_pg/etc/query.conf,${project_directory}/etc/collectors/ mail/rtec/tools/shards_analyzer.py,${project_directory}/deploy/usr/share/collectors/'
])

create_job('b_xiva_dba', [
  'name':'xiva-dba',
  'project_directory':'mail/xiva/dba',
  'project_binary': 'xivadba',
  'SKIP_UNIT_TESTS': '1',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_mailpusher', [
  'name':'mailpusher',
  'project_directory':'mail/xiva/mailpusher',
  'project_binary': 'mailpusher',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_xiva_crm', [
  'name':'xivacrm',
  'project_directory':'mail/xiva/crm',
  'project_binary': 'xivacrm',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_leasemeta', [
  'name':'leasemeta',
  'project_directory':'mail/leasemeta',
  'project_binary': 'leasemeta',
  'SKIP_UNIT_TESTS': '1',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_httpclient_stress', [
  'name':'httpclient',
  'project_directory':'mail/ymod_httpclient/examples/stress',
  'project_binary': 'yplatform',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_webserver_stress', [
  'name':'webserver',
  'project_directory':'mail/ymod_webserver/examples/stress',
  'project_binary': 'yplatform',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_rtec_duty', [
  'name':'rtec-duty',
  'project_directory':'mail/rtec/duty',
  'project_binary': 'rtec',
  'SKIP_UNIT_TESTS': '1',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_collectors_ext', [
  'name':'collectors-ext',
  'project_directory':'mail/collectors-ext',
  'project_binary': 'collectors-ext',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_collectors_ext_monitor', [
  'name':'collectors-ext-monitor',
  'project_directory':'mail/collectors-ext/monitoring',
  'project_binary': 'collectors-ext',
  'paths_to_copy': 'mail/collectors-ext/etc/collectors-ext/shards,${project_directory}/etc/collectors-ext/',
  'SKIP_UNIT_TESTS': '1',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_xiva_dbstats', [
  'name':'xiva-dbstats',
  'project_directory':'mail/xiva/dbstats',
  'project_binary': 'xivadb-stats',
  'paths_to_copy': 'mail/xiva/dba/src/shards.py,${project_directory}/src/ mail/xiva/dba/src/db_requests.py,${project_directory}/src/',
  'SKIP_UNIT_TESTS': '1',
  'SKIP_SYSTEM_TESTS': '1'
])


create_job('b_botserver', [
  'name':'botserver',
  'project_directory':'mail/botserver',
  'project_binary': 'botserver'
])


create_job('b_fan_send', [
  'name':'fan-send',
  'project_directory':'mail/fan/fan_send',
  'project_binary': 'fan_send',
  'SKIP_SYSTEM_TESTS': '1'
])

create_job('b_tractor', [
    'name': 'tractor',
    'project_directory': 'mail/tractor',
    'paths_to_copy': 'mail/tractor/src/*,destdir/app/python/ mail/tractor/requirements.txt,destdir/app/python/',
    'SKIP_BUILD': '1',
], 'deploy')
