// **************
// HELPFUL LINKS
// **************
// https://jenkinsci.github.io/job-dsl-plugin
// https://github.yandex-team.ru/qatools/startrek-jenkins-plugin/blob/fa16a72deed91b3cab7493717a87156f63076202/README.md

def create_job = { name, envs, desc ->
freeStyleJob(name) {
  description("""
Сборка докер-образов для сборки проектов
<br/>
Эта джоба сгенерирована автоматически. <br/>
Подробности смотреть в <a href=${JOB_URL}>Seed Job</a>
<br/>""")

  label('xiva_deb')
  logRotator(14, 200)
  concurrentBuild(false)
  parameters {
      stringParam('BRANCH', '', 'ветка в репозитории, которую собирать')
      booleanParam('CLEAN_WORKING_DIR', false, 'снести сборочную директорию')
      booleanParam('RUN_DOCKER_CLEAN', false, 'снести все ранее собранные образы')
      booleanParam('SKIP_TESTS', false, 'пропустить прогон тестов')
      booleanParam('SKIP_PUBLISHING', false, 'пропустить публикацию докер образов')
  }
  environmentVariables(envs)

  wrappers {
      colorizeOutput('xterm')
      sshAgent('robot-gerrit-ssh-key')

      credentialsBinding {
          usernamePassword('DOCKER_REGISTRY_USER', 'DOCKER_REGISTRY_TOKEN', 'robot-sender-docker-registry-creds')
      }

      preBuildCleanup {
          deleteDirectories()
          cleanupParameter('CLEAN_WORKING_DIR')
      }

      timeout {
          absolute(minutes = 60)
      }
  }

  scm {
      git {
          remote {
              url('git@bb.yandex-team.ru:mail/arcadia.git')
              credentials('robot-gerrit-ssh-key')
          }
          branch('${BRANCH}')
          extensions {
              // dot not shallow clone because shallow updates are not allowed in git repos
              submoduleOptions {
                  recursive()
              }
          }
      }
  }
  steps {
      shell('''
        set -eo pipefail
        DT=`date "+%Y-%m-%d-%H-%M"`;
        VERSION="${DT}${BRANCH}"
        BUILDNAME="${name}:r${VERSION}"
        echo "VERSION=${VERSION}" > variables.txt
        echo "GITTAG=fan-${name}-${VERSION}" >> variables.txt
        echo "BUILDNAME=${BUILDNAME}" >> variables.txt
        echo "BUILDTAG=registry.yandex.net/mail/fan/fan-${BUILDNAME}" >> variables.txt
        echo "BUILDARGS=--checkout -j`nproc` --build=relwithdebinfo" >> variables.txt
      ''')

      environmentVariables {
          propertiesFile('variables.txt')
      }

      shell('''
        set -eo pipefail
        [[ -d ~/asg ]] && ~/asg/asg selfupdate
        [[ -d ~/asg ]] || git clone --depth 1 https://github.yandex-team.ru/xiva/asg ~/asg && ~/asg/asg-integrate-bash
      ''')

      shell('''
        set -eo pipefail
        [[ -d build ]] || ~/asg/asg clone --branch ${BRANCH} build
        cd build
        cd mail
        git fetch
        git reset --hard
        git clean -dfx
        git checkout origin/${BRANCH}
        cd ..
        ~/asg/asg up
      ''')

      shell('''
        set -eo pipefail
        cd build
        cd mail/fan
        [[ "${RUN_DOCKER_CLEAN}" == "true" ]] && make docker-clean
        make docker-build-${name}
      ''')

      shell('''
        set -eo pipefail
        [[ "${SKIP_TESTS}" == "true" ]] && { echo "skip tests"; exit 0; }
        cd build
        cd mail/fan
        make test-${name}
      ''')

      shell('''
        set -eo pipefail
        [[ "${SKIP_PUBLISHING}" == "true" ]] && { echo "skip publishing"; exit 0; }
        pushd build
        cd mail/fan
        make VERSION="r${VERSION}" docker-tag-${name}
        make VERSION="r${VERSION}" docker-push-${name}
        popd
        DIGEST=$(docker inspect -f '{{.RepoDigests}}' ${BUILDTAG} | awk -F '(\\\\[|@|\\\\])' '{print $3}')
        echo "DIGEST=$DIGEST" >> variables.txt
      ''')

      environmentVariables {
          propertiesFile('variables.txt')
      }

      publishers {
          postBuildTask {
              task('.*', '''
                  # cleanup docker containers and images after build
                  docker stop $(docker ps -a -q) || true # do not fail if no running containers left
                  docker container prune --force
                  docker image rm ${BUILDTAG} || true # do not fail if image was not build successfully
                  docker image prune --force
                  docker volume prune --force
              ''')
          }
      }

      publishers {
          git {
              pushOnlyIfSuccess()
              tag('origin', '${GITTAG}') {
                  create(true)
                  update(true)
              }
          }
      }

      publishers {
          buildDescription('', '''${BUILDTAG} ${DIGEST}''')
      }
  }

  wrappers {
    buildName('${BUILDNAME}')
  }

}
}

create_job('fan_ui', [
  'name':'ui'
], '')

create_job('fan_feedback', [
  'name':'feedback'
], '')

create_job('fan_api', [
  'name':'api'
], '')

create_job('fan_smtp-mock', [
  'name':'smtp-mock'
], '')

create_job('fan_mocks', [
  'name':'mocks'
], '')
