// **************
// HELPFUL LINKS
// **************
// https://jenkinsci.github.io/job-dsl-plugin
// https://github.yandex-team.ru/qatools/startrek-jenkins-plugin/blob/fa16a72deed91b3cab7493717a87156f63076202/README.md

def create_job = { job_name, envs, cloud = 'qloud' ->
freeStyleJob(job_name) {
  description("""
Сборка докер-образов для сборки проектов
<br/>
Эта джоба сгенерирована автоматически. <br/>
Подробности смотреть в <a href=${JOB_URL}>Seed Job</a>
<br/>""")

  label('xiva_deb')
  logRotator(14, 200)
  concurrentBuild(false)
  parameters {
      stringParam('BRANCH', 'master', 'ветка в репозитории, которую собирать')
      booleanParam('CLEAN_WORKING_DIR', false, 'снести сборочную директорию')
  }
  environmentVariables(envs)

  wrappers {
      colorizeOutput('xterm')
      sshAgent('robot-gerrit-ssh-key')

      credentialsBinding {
          usernamePassword('DOCKER_REGISTRY_USER', 'DOCKER_REGISTRY_TOKEN', 'robot-sender-docker-registry-creds')
      }

      preBuildCleanup {
          deleteDirectories()
          cleanupParameter('CLEAN_WORKING_DIR')
      }
  }

  scm {
      git {
          remote {
              url('git@bb.yandex-team.ru:mail/arcadia.git')
              branch('${BRANCH}')
              credentials('robot-gerrit-ssh-key')
          }
          branch('${BRANCH}')
          extensions {
              // dot not shallow clone because shallow updates are not allowed in git repos
              submoduleOptions {
                  recursive()
              }
          }
      }
  }
  steps {
      shell('''
        set -eo pipefail
        DT=`date "+%Y-%m-%d-%H-%M"`;
        VERSION="${DT}${BRANCH}"
        BUILDNAME="${name}:r${VERSION}"
        echo "VERSION=${VERSION}" > variables.txt
        echo "GITTAG=${name}-${VERSION}" >> variables.txt
        echo "BUILDNAME=${BUILDNAME}" >> variables.txt
        echo "BUILDTAG=registry.yandex.net/mail/xiva/${BUILDNAME}" >> variables.txt
        echo "BUILDARGS=--checkout -j`nproc --ignore=10` --build=relwithdebinfo" >> variables.txt
      ''')

      environmentVariables {
          propertiesFile('variables.txt')
      }

      shell('''
        set -eo pipefail
        [ -d ~/asg ] && ~/asg/asg selfupdate
        [ -d ~/asg ] || git clone --depth 1 https://github.yandex-team.ru/xiva/asg ~/asg && ~/asg/asg-integrate-bash
      ''')

      shell('''
        set -eo pipefail
        [ -d build ] || ~/asg/asg clone --branch ${BRANCH} build
        cd build
        cd mail
        git fetch
        git reset --hard
        git clean -dfx
        git checkout origin/${BRANCH}
        cd ..
        ~/asg/asg up
      ''')

      shell('''
        set -eo pipefail
        [ "${SKIP_BUILD}" == "1" ] && { echo "skip build"; exit 0; }
        cd build
        ~/asg/asg make "${BUILDARGS}" "${project_directory}" mail/yplatform/config-dumper/ mail/pa/
      ''')

      shell('''
        set -eo pipefail
        [ "${SKIP_UNIT_TESTS}" == "1" ] && { echo "skip tests"; exit 0; }
        cd build
        ~/asg/asg make "${BUILDARGS}" -t "${project_directory}"
      ''')

      shell('''
        set -eo pipefail
        [ "${SKIP_SYSTEM_TESTS}" == "1" ] && { echo "skip system tests"; exit 0; }
        cd build
        cd "${project_directory}/tests-system"
        if [ -f "./run.sh" ]; then
            BUILD_ROOT_DIR="../" sh ./run.sh || BUILD_ROOT_DIR="../" sh ./run.sh
        else
            ~/asg/asg make "${BUILDARGS}" -t --test-size=LARGE
        fi
      ''')

      def prepare_destdir_steps = '''
        copy() {
            src="$1"
            dst="$2"
            mkdir -p "$dst"
            cp -r "$src" "$dst"
        }

        cd build
        rm -rf destdir
        cp -r mail/rtec/docker-common-deploy/deploy "${project_directory}/${name}"
        mkdir destdir
        mkdir -p destdir/usr/bin/
        mkdir -p destdir/usr/sbin/
        cp ya destdir/usr/bin/
        for i in ${paths_to_copy[@]}; do copy $(echo $i | tr "," " "); done
      '''

      def copy_common_binaries_steps = '''
        cp mail/yplatform/config-dumper/config-dumper destdir/usr/bin/
        cp mail/pa/pa/pa destdir/usr/bin/
      '''

      def copy_config_step = '''
        cp -Lr "\${project_directory}/${name}/config" destdir/app/config/
      '''

      if (cloud == 'deploy') {
          shell("""
            set -eo pipefail
            ${prepare_destdir_steps}
            mkdir -p destdir/app/
            ${copy_config_step}

            [ "\${SKIP_BUILD}" == "1" ] && { echo "build skipped, no binaries copied"; exit 0; }
            cp "\${project_directory}/\${project_binary}" destdir/app/
            ${copy_common_binaries_steps}
        """)
      } else {
          throw new Exception("Unknown cloud: ${cloud}")
      }

      shell('''
        set -eo pipefail
        pushd build
        docker build --pull --network=host --build-arg installdir="destdir" --build-arg deploydir="${project_directory}/${name}/deploy" --tag ${BUILDTAG} -f ${project_directory}/${name}/Dockerfile .
        docker push ${BUILDTAG}
        popd
        DIGEST=$(docker inspect -f '{{.RepoDigests}}' ${BUILDTAG} | awk -F '(\\\\[|@|\\\\])' '{print $3}')
        echo "DIGEST=$DIGEST" >> variables.txt
      ''')

      environmentVariables {
          propertiesFile('variables.txt')
      }

      publishers {
          postBuildTask {
              task('.*', '''
                  # cleanup docker containers and images after build
                  docker stop $(docker ps -a -q) || true # do not fail if no running containers left
                  docker rm -f $(docker ps -a -q) || true
                  docker container prune --force
                  docker image rm ${BUILDTAG} || true # do not fail if image was not build successfully
                  docker image prune --force
                  docker volume prune --force
              ''')
          }
      }

      publishers {
          git {
              pushOnlyIfSuccess()
              tag('origin', '${GITTAG}') {
                  create(true)
                  update(true)
              }
          }
      }

      publishers {
          buildDescription('', '''${BUILDTAG} ${DIGEST}''')
      }
  }

  wrappers {
    buildName('${BUILDNAME}')
  }

}
}

create_job('b_tractor_api', [
    'name': 'tractor_api',
    'project_directory': 'mail/tractor',
    'paths_to_copy': 'mail/tractor/pylib/*,destdir/app/python/ mail/tractor/tractor_api/src/*,destdir/app/python/  mail/tractor/requirements/base.txt,destdir/app/requirements/ mail/tractor/requirements/tractor_api.txt,destdir/app/requirements/',
    'SKIP_BUILD': '1',
    'SKIP_UNIT_TESTS': '1',
    'SKIP_SYSTEM_TESTS': '1',
], 'deploy')

create_job('b_tractor_disk', [
    'name': 'tractor_disk',
    'project_directory': 'mail/tractor',
    'paths_to_copy': 'mail/tractor/pylib/*,destdir/app/python/ mail/tractor/tractor_disk/src/*,destdir/app/python/ mail/tractor/tractor_disk/config,destdir/app/  mail/tractor/requirements/base.txt,destdir/app/requirements/ mail/tractor/requirements/tractor_disk.txt,destdir/app/requirements/',
    'SKIP_BUILD': '1',
    'SKIP_UNIT_TESTS': '1',
    'SKIP_SYSTEM_TESTS': '1',
], 'deploy')

create_job('b_tractor_mail', [
    'name': 'tractor_mail',
    'project_directory': 'mail/tractor',
    'paths_to_copy': 'mail/tractor/pylib/*,destdir/app/python/ mail/tractor/tractor_mail/src/*,destdir/app/python/ mail/tractor/tractor_mail/config,destdir/app/  mail/tractor/requirements/base.txt,destdir/app/requirements/ mail/tractor/requirements/tractor_mail.txt,destdir/app/requirements/',
    'SKIP_BUILD': '1',
    'SKIP_UNIT_TESTS': '1',
    'SKIP_SYSTEM_TESTS': '1',
], 'deploy')
