#!/bin/bash -ex

if [[ -z $SOURCE_BRANCH ]]; then
    SOURCE_BRANCH="origin/master"
fi

COMMIT_INTERVAL=60
PREV_SVN_COMMIT_TIME=0

arcadia_clone () {
    echo "Cloning ya via SVN..."
    svn export -q svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya
    YA=${PWD}/ya
    export YA
    echo "Cloning arcadia via SVN..."
    ${YA} clone -q arcadia
    cd arcadia
    echo "Cloning ozo..."
    ${YA} make -k -q --checkout -j0 mail/ozo
    OZO_DIR=${PWD}/mail/ozo
    GITHUB_DIR=$OZO_DIR/github
}

github_clone () {
    cd ${GITHUB_DIR}
    git init
    git remote add origin $(cat .url)
    git remote update
    REMOTE_ID=$(git rev-parse ${SOURCE_BRANCH})
    LOCAL_ID=$(get_local_id)
}

git_checkout () {
    local FLAGS=$([[ ${OVERWRITE_ARCADIA_CHANGES} == 'true' ]] && echo "-f" || echo)
    ${OZO_DIR}/git_checkout.sh -d "${OZO_DIR}" $FLAGS $1
}

get_local_id () {
    cat ${GITHUB_DIR}/.id
}

check_for_arcadia_changes () {
    cd ${GITHUB_DIR}
    git checkout -f ${LOCAL_ID}

    if [[ ${OVERWRITE_ARCADIA_CHANGES} != 'true' ]] && [[ $(svn diff . | wc -l) -gt 0 ]]; then
        echo 'Repository contains changes are not commited into github'
        exit -1
    fi
}

check_for_sync () {
    if [[ "${1}" == "" ]]; then
        if [[ "${LOCAL_ID}" != "${REMOTE_ID}" ]]; then
            echo "${SOURCE_BRANCH} can not be syncronized with ${LOCAL_ID}: no fast forward path"
            exit 1
        fi
        echo 'Repository is synchronized'
        exit 0
    fi
}

check_ci () {
    local RESULT=0
    ${OZO_DIR}/check_ci.sh -d ${OZO_DIR} || RESULT=$?
    if [ $RESULT == 200 ]; then
        echo "CI checks failed"
    fi
    return $RESULT
}

svn_prepare () {
    cd ${GITHUB_DIR}
    local FILES_TO_DELETE=$(svn status | grep '^!' | sed 's;!;;')
    echo "Files to delete: ${FILES_TO_DELETE}"
    if [[ "${FILES_TO_DELETE}" != "" ]]; then
        svn rm --force -q ${FILES_TO_DELETE} || true
    fi
    # Files to add = (git XOR svn) AND git
    local FILES_TO_ADD=$(echo $(echo $(git ls-files) $(svn ls --recursive) | tr ' ' '\n' | sort | uniq -u) $(git ls-files) | tr ' ' '\n' | sort | uniq -d)
    echo "Files to add: ${FILES_TO_ADD}"
    if [[ "${FILES_TO_ADD}" != "" ]]; then
        svn add --parents -q ${FILES_TO_ADD} || true
    fi
}

svn_commit () {
    cd ${GITHUB_DIR}
    printf "$(git log --format=%B -n 1)\n\n$(git log --format='%H %aN <%aE>' -n 1)\n" > .message
    cd ${OZO_DIR}
    NOW=$(date +%s)
    if [[ $(( NOW - PREV_SVN_COMMIT_TIME )) -lt ${COMMIT_INTERVAL} ]] && [[ $(( NOW - PREV_SVN_COMMIT_TIME )) -ge 0 ]]; then
        sleep $(( COMMIT_INTERVAL - NOW + PREV_SVN_COMMIT_TIME ))
    fi
    svn commit \
        --with-revprop arcanum:force=yes \
        --with-revprop arcanum:review-skip=yes \
        --with-revprop arcanum:check-skip=yes \
        -F github/.message \
        .
    PREV_SVN_COMMIT_TIME=$(date +%s)
}

get_commits () {
    cd ${GITHUB_DIR};echo $(git log ${LOCAL_ID}..${REMOTE_ID} --pretty=%H $1)
}

# Apply function for each of commit from an array or exit if function returns
# code not equal to specified.
#   $1 - array of commits
#   $2 - function to apply
#   $3 - desired return code
for_each_commit () {
    local RESULT=0
    for COMMIT in $1; do
        git_checkout ${COMMIT}
        svn_prepare
        $2 || RESULT=$?
        if [ $RESULT != $3 ]; then
            break
        fi
    done
    return $RESULT
}

arcadia_clone
github_clone

COMMITS_TO_CHECK=$(get_commits)
check_for_sync "${COMMITS_TO_CHECK}"
check_for_arcadia_changes

for_each_commit "${COMMITS_TO_CHECK}" check_ci 200

COMMITS_TO_SYNC=$(get_commits --reverse)
for_each_commit "${COMMITS_TO_SYNC}" svn_commit 0
