from copy import deepcopy

from juggler_sdk import Check, Child, FlapOptions


AGG_NAME = "more_than_limit_is_problem"
AGG_KWARGS = {
    "mode": "percent",
    "nodata_mode": "force_crit",
    "warn_limit": "1",
    "crit_limit": "20",
}

FLAPS_CONF = FlapOptions(
    stable=300,
    critical=1500,
    boost=0,
)


def get_meta(env):
    return {
        "urls": [{
            "url": "https://platform.yandex-team.ru/projects/mail/ava/{0}".format(env),
            "type": "qloud",
            "title": "Qloud environment url",
        }]
    }


def get_tags(env):
    return [
        "qloud-ext.mail",
        "qloud-ext.mail.ava",
        "qloud-ext.mail.ava.{}".format(env),
        "mail.ava",
        "maildev",
        "mail",
        "env_{}".format(env),
    ]


def get_check(service, env, comp, active=None, active_kwargs=None, pass_unreachable=True):
    meta = get_meta(env)
    children = [Child(
        host="mail.ava.{}.{}@type=ext".format(env, comp),
        service=service,
        group_type="QLOUD"
    )]
    tags = get_tags(env)
    agg_kwargs = deepcopy(AGG_KWARGS)
    if pass_unreachable:
        agg_kwargs['unreach_mode'] = 'force_ok'
        agg_kwargs['unreach_service'] = [{'check': ':UNREACHABLE'}]
    return Check(
        namespace="mail.maildev",
        host="ava-{}".format(comp),
        service=service,
        refresh_time=60,
        ttl=1500,
        aggregator=AGG_NAME,
        aggregator_kwargs=agg_kwargs,
        flaps_config=FLAPS_CONF,
        tags=tags,
        meta=meta,
        children=children,
        active=active,
        active_kwargs=active_kwargs,
    )


def get_unreachable(env, comp):
    check = get_check('UNREACHABLE', env, comp, active='icmpping', active_kwargs={}, pass_unreachable=False)
    check.aggregator_kwargs = {
        "mode": "percent",
        "nodata_mode": "force_crit",
        "warn_limit": "35",
        "crit_limit": "35",
    }
    return check


def get_4xx(env, comp):
    return get_check('ava_4xx', env, comp)


def get_5xx(env, comp):
    return get_check('ava_5xx', env, comp)


def get_unispace(env, comp):
    return get_check('unispace', env, comp)


def get_tcp_check(env, comp):
    active_kwargs = {
        "disable_ipv4": True,
        "port": 80,
        "timeout": 5,
    }
    return get_check('default_tcp_check', env, comp, active='tcp_chat', active_kwargs=active_kwargs)


def get_checks():
    components = [
        ("production", "canary"),
        ("production", "ava")
    ]
    for component in components:
        yield get_unreachable(component[0], component[1])
        yield get_tcp_check(component[0], component[1])
        yield get_5xx(component[0], component[1])
        yield get_4xx(component[0], component[1])
        yield get_unispace(component[0], component[1])
