from collections import namedtuple
from copy import deepcopy

from juggler_sdk import FlapOptions, Child, Check

JugglerDeployDescription = namedtuple("JugglerDeployDescription", "environment, component, host")
AGG_NAME = "more_than_limit_is_problem"
AGG_KWARGS = {
    "mode": "percent",
    "nodata_mode": "force_crit",
    "warn_limit": "1",
    "crit_limit": "20",
}
FLAPS_CONF = FlapOptions(
    stable=300,
    critical=1500,
    boost=0,
)


def get_meta(env):
    return {
        "urls": [{
            "url": "https://platform.yandex-team.ru/projects/mail/reminders/{}".format(env),
            "type": "qloud",
            "title": "Qloud environment url",
        }]
    }


def get_tags(env):
    return [
        "qloud-ext.mail",
        "qloud-ext.mail.reminders",
        "qloud-ext.mail.reminders.{}".format(env),
        "mail.reminders",
        "maildev",
        "mail",
        "env_{}".format(env)
    ]


def get_check(service, description, active=None, active_kwargs=None, agg_kwargs=None, unreach_services=tuple()):
    meta = get_meta(description.environment)
    children = [Child(
        host="mail.reminders.{}.{}@type=ext".format(description.environment, description.component),
        service=service,
        group_type="QLOUD"
    )]
    tags = get_tags(description.environment)
    agg_kwargs = agg_kwargs or deepcopy(AGG_KWARGS)
    if unreach_services:
        agg_kwargs['unreach_mode'] = 'force_ok'
        agg_kwargs['unreach_service'] = [{'check': unreach} for unreach in unreach_services]
    check = Check(
        namespace="mail.reminders",
        host=description.host,
        service=service,
        refresh_time=60,
        ttl=1500,
        aggregator=AGG_NAME,
        aggregator_kwargs=agg_kwargs,
        flaps_config=FLAPS_CONF,
        tags=tags,
        meta=meta,
        children=children,
        active=active,
        active_kwargs=active_kwargs,
    )
    return check


def get_unreachable(description):
    check = get_check(
        'UNREACHABLE',
        description,
        active='icmpping',
        active_kwargs={},
        agg_kwargs={
            "mode": "percent",
            "nodata_mode": "force_crit",
            "warn_limit": "35",
            "crit_limit": "35",
        }
    )
    return check


def get_tcp_check(description):
    return get_check(
        'default_tcp_check',
        description,
        active='tcp_chat',
        active_kwargs={
            "disable_ipv4": True,
            "port": 80,
            "timeout": 5,
        },
        agg_kwargs={
            "mode": "percent",
            "nodata_mode": "force_crit",
            "warn_limit": "35",
            "crit_limit": "35",
        },
        unreach_services=(':UNREACHABLE',)
    )


def get_active_http(description, name, path):
    active_kwargs = {
        "disable_ipv4": True,
        "ok_codes": [],
        "headers": {},
        "timeout": 5,
        "warn_codes": [],
        "path": path,
        "port": 80
    }
    return get_check(name, description, active="http", active_kwargs=active_kwargs,
                     unreach_services=(':UNREACHABLE', ':default_tcp_check'))


def get_checks():
    descriptions = [
        JugglerDeployDescription(environment="production", component="api", host="reminders-api"),
        JugglerDeployDescription(environment="production", component="worker", host="reminders-worker")
    ]

    for description in descriptions:
        yield get_unreachable(description)
        yield get_tcp_check(description)
        yield get_active_http(description, "ping", "/ping")
        yield get_active_http(description, "ping-mongo", "/ping-mongo")
        yield get_active_http(description, "ping-alive-apps", "/ping-alive-apps")
