package ru.yandex.kitsune;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.logging.Level;

import javax.annotation.Nonnull;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.RequestLine;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.HttpResponseSendingCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.DuplexFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.PayloadFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncResponseProducerGenerator;
import ru.yandex.http.util.nio.EntityGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityGeneratorAsyncConsumer;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.kitsune.config.ImmutableKitsuneConfig;
import ru.yandex.kitsune.config.ImmutableProxyHttpHostConfig;

public class KitsuneHttpServer
    extends HttpProxy<ImmutableKitsuneConfig>
    implements HttpAsyncRequestHandler<EntityGenerator>
{
    private final ClientWithConfig<ImmutableHttpHostConfig> head;

    public KitsuneHttpServer(@Nonnull ImmutableKitsuneConfig config) throws IOException {
        super(config);

        head = clientWithConfig("Head", config.head());

        final Map<String, ClientWithConfig<ImmutableProxyHttpHostConfig>> tails;
        if (!config.tails().isEmpty()) {
            tails = new HashMap<>(config.tails().size());
            for (var entry : config.tails().entrySet()) {
                final String name = entry.getKey();
                tails.put(name, clientWithConfig(name, entry.getValue()));
            }
        } else {
            tails = Collections.emptyMap();
        }

        for (var entry : config.proxyPatterns().entrySet()) {

            final Map<String, ClientWithConfig<ImmutableProxyHttpHostConfig>> tailsMatchedPattern =
                    new HashMap<>(tails.size());

            for (var tailEntry : tails.entrySet()) {
                final ClientWithConfig<ImmutableProxyHttpHostConfig> tail = tailEntry.getValue();
                if (tail.config().proxyPatterns().contains(entry.getKey())) {
                    tailsMatchedPattern.put(tailEntry.getKey(), tail);
                }
            }

            final HttpAsyncRequestHandler<EntityGenerator> handler;
            if (tailsMatchedPattern.isEmpty()) {
                logger.warning("zero tails for " + entry.getKey());
                handler = this;
            } else {
                handler = new Handler(tailsMatchedPattern);
            }
            register(entry.getValue(), handler);
        }
    }

    public void compare(@Nonnull ProxySession session, @Nonnull BasicAsyncResponseProducerGenerator head,
                        @Nonnull List<Map.Entry<String, HttpResponse>> tails) {

    }

    @Nonnull
    private static BasicAsyncRequestProducerGenerator makeRequest(@Nonnull ProxySession session,
                                                                  @Nonnull EntityGenerator entityGenerator) {
        final RequestLine requestLine = session.request().getRequestLine();
        String method = requestLine.getMethod();

        final BasicAsyncRequestProducerGenerator requestProducerGenerator =
                new BasicAsyncRequestProducerGenerator(requestLine.getUri(), entityGenerator, method);

        requestProducerGenerator.copyHeader(session.request(), HttpHeaders.CONTENT_TYPE);
        requestProducerGenerator.copyHeader(session.request(), HttpHeaders.ACCEPT_ENCODING);
        requestProducerGenerator.copyHeader(session.request(), HttpHeaders.CONTENT_ENCODING);
        requestProducerGenerator.copyHeader(session.request(), "Content-Disposition");

        return requestProducerGenerator;
    }

    private class Handler implements HttpAsyncRequestHandler<EntityGenerator> {
        @Nonnull
        private final Map<String, ClientWithConfig<ImmutableProxyHttpHostConfig>> tails;

        private Handler(@Nonnull Map<String, ClientWithConfig<ImmutableProxyHttpHostConfig>> tails) {
            this.tails = tails;
        }

        @Override
        public HttpAsyncRequestConsumer<EntityGenerator> processRequest(HttpRequest httpRequest,
                                                                        HttpContext httpContext) {
            return new NByteArrayEntityGeneratorAsyncConsumer();
        }

        @Override
        public void handle(EntityGenerator entityGenerator, HttpAsyncExchange exchange, HttpContext context) throws HttpException {
            final ProxySession session = new BasicProxySession(KitsuneHttpServer.this, exchange, context);

            final DoubleFutureCallback<BasicAsyncResponseProducerGenerator, List<Map.Entry<String, HttpResponse>>> headTailsCallbacks = new DoubleFutureCallback<>(new CompareCallback(session));

            head.execute(
                    session,
                    makeRequest(session, entityGenerator),
                    new ResponseProducerGeneratorCallback(
                            new DuplexFutureCallback<>(
                                    headTailsCallbacks.first(),
                                    new HttpResponseGeneratorSendingCallback(session))));

            final MultiFutureCallback<Map.Entry<String, HttpResponse>> tailsCallbacks =
                    new MultiFutureCallback<>(headTailsCallbacks.second());
            try {
                final Random randomGenerator = new Random();
                for (var tailEntry : tails.entrySet()) {
                    final String tailName = tailEntry.getKey();
                    final ClientWithConfig<ImmutableProxyHttpHostConfig> tail = tailEntry.getValue();

                    final boolean forcingRequest = session.params().getBoolean(tailName, false);

                    if (forcingRequest || randomGenerator.nextDouble() < tail.config().probability()) {
                        tail.execute(
                                session,
                                makeRequest(session, entityGenerator),
                                new ErrorSuppressingFutureCallback<>(
                                        new PayloadFutureCallback<>(
                                                tailEntry.getKey(),
                                                tailsCallbacks.newCallback()),
                                        (HttpResponse) null));
                    }
                }
            } finally {
                tailsCallbacks.done();
            }
        }
    }

    @Override
    public HttpAsyncRequestConsumer<EntityGenerator> processRequest(HttpRequest httpRequest, HttpContext httpContext) {
        return new NByteArrayEntityGeneratorAsyncConsumer();
    }

    @Override
    public void handle(EntityGenerator entityGenerator, HttpAsyncExchange exchange, HttpContext context) throws HttpException {
        final ProxySession session = new BasicProxySession(this, exchange, context);
        head.execute(session, makeRequest(session, entityGenerator), new HttpResponseSendingCallback(session));
    }

    public <T extends ImmutableHttpHostConfig> ClientWithConfig<T> clientWithConfig(final String name, final T config) {
        final AsyncClient client = new AsyncClient(reactor, config, RequestErrorType.ERROR_CLASSIFIER);
        return new ClientWithConfig<>(registerClient(name + "Client", client, config), config);
    }

    public class CompareCallback extends AbstractProxySessionCallback<Map.Entry<BasicAsyncResponseProducerGenerator,
            List<Map.Entry<String, HttpResponse>>>> {
        protected CompareCallback(ProxySession session) {
            super(session);
        }

        @Override
        public void completed(Map.Entry<BasicAsyncResponseProducerGenerator, List<Map.Entry<String, HttpResponse>>> httpEntityListMap) {
            try {
                compare(session, httpEntityListMap.getKey(), httpEntityListMap.getValue());
            } catch (Exception e) {
                session.logger().log(Level.WARNING, "comparing failed", e);
            }
        }
    }

    public static class ResponseProducerGeneratorCallback extends AbstractFilterFutureCallback<HttpResponse,
            BasicAsyncResponseProducerGenerator> {
        public ResponseProducerGeneratorCallback(FutureCallback<? super BasicAsyncResponseProducerGenerator> callback) {
            super(callback);
        }

        @Override
        public void completed(HttpResponse httpResponse) {
            try {
                callback.completed(new BasicAsyncResponseProducerGenerator(httpResponse));
            } catch (IOException e) {
                failed(e);
            }
        }
    }

    public static class HttpResponseGeneratorSendingCallback extends AbstractProxySessionCallback<BasicAsyncResponseProducerGenerator> {
        public HttpResponseGeneratorSendingCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final BasicAsyncResponseProducerGenerator response) {
            session.response(response.get());
        }
    }

    public static class ClientWithConfig<T extends ImmutableHttpHostConfig> {
        @Nonnull
        private final AsyncClient client;
        @Nonnull
        private final T config;

        private ClientWithConfig(@Nonnull AsyncClient client, @Nonnull T config) {
            this.client = client;
            this.config = config;
        }


        public void execute(@Nonnull ProxySession session,
                            @Nonnull BasicAsyncRequestProducerGenerator producerGenerator,
                            @Nonnull FutureCallback<HttpResponse> callback) {
            try {
                final AsyncClient client = this.client.adjust(session.context());
                client.execute(config.host(), producerGenerator, BasicAsyncResponseConsumerFactory.INSTANCE,
                        session.listener().createContextGeneratorFor(client), callback);
            } catch (RuntimeException e) {
                callback.failed(e);
            }
        }

        @Nonnull
        public final T config() {
            return config;
        }
    }
}

