package ru.yandex.kitsune.config;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

import ru.yandex.collection.Pattern;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.request.RequestPatternParser;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractKitsuneConfigBuilder
        <T extends AbstractKitsuneConfigBuilder<T>>
        extends AbstractHttpProxyConfigBuilder<T>
        implements KitsuneConfig {
    @Nonnull
    private HttpHostConfigBuilder head;

    @Nonnull
    private Map<String, ProxyHttpHostConfigBuilder> tails;

    @Nonnull
    private Map<String, Pattern<RequestInfo>> proxyPatterns;

    protected AbstractKitsuneConfigBuilder(final KitsuneConfig config) {
        super(config);

        head = new HttpHostConfigBuilder(config.head());

        tails(config.tails());

        proxyPatterns = config.proxyPatterns();
    }

    protected AbstractKitsuneConfigBuilder(
            final IniConfig config,
            final KitsuneConfig defaults)
            throws ConfigException, IOException {
        super(config, defaults);

        {
            final IniConfig section = config.section("head");
            head = new HttpHostConfigBuilder(section, defaults.head());
        }

        {
            final IniConfig section = config.section("tails");
            tails = new LinkedHashMap<>(2 * config.sections().size());

            for (var entry : section.sections().entrySet()) {
                final ProxyHttpHostConfig defaultConfig = defaults.tails().getOrDefault(entry.getKey(), null);
                tails.put(entry.getKey(), new ProxyHttpHostConfigBuilder(
                        entry.getValue(),
                        defaultConfig != null ? defaultConfig : ProxyHttpHostConfigDefaults.INSTANCE));
            }
        }

        {
            final IniConfig proxyPatternSection = config.section("proxy-patterns");
            final Set<String> rawPatternsKeys = proxyPatternSection.keys();
            if (!rawPatternsKeys.isEmpty()) {
                proxyPatterns = new LinkedHashMap<>(2 * rawPatternsKeys.size());
                for (String key : rawPatternsKeys) {
                    proxyPatterns.put(
                            key,
                            proxyPatternSection.get(key, RequestPatternParser.INSTANCE));
                }
            } else {
                proxyPatterns = Collections.emptyMap();
            }
        }
    }

    @Nonnull
    @Override
    public HttpHostConfigBuilder head() {
        return head;
    }

    public void head(HttpHostConfigBuilder head) {
        this.head = head;
    }

    @Nonnull
    @Override
    public Map<String, ProxyHttpHostConfigBuilder> tails() {
        return tails;
    }

    public void tails(@Nonnull Map<String, ? extends ProxyHttpHostConfig> tails) {
        this.tails = new LinkedHashMap<>(2 * tails.size());

        for (var entry : tails.entrySet()) {
            this.tails.put(entry.getKey(), new ProxyHttpHostConfigBuilder(entry.getValue()));
        }
    }

    @Nonnull
    @Override
    public Map<String, Pattern<RequestInfo>> proxyPatterns() {
        return proxyPatterns;
    }

    public void proxyPatterns(@Nonnull Map<String, Pattern<RequestInfo>> proxyPatterns) {
        this.proxyPatterns = proxyPatterns;
    }
}

