#pragma once
//  accessors.h
//
// Copyright (c) Yandex, LLC, 2010. All rights reserved.
//
// Provides response data accessors for the BlackBox HTTP/XML API

#ifndef BLACKBOX_ACCESSORS_H_INCLUDED
#define BLACKBOX_ACCESSORS_H_INCLUDED

#include <string>
#include <memory>
#include <vector>
#include <map>
#include <stdint.h>

namespace bb {

    /// Default argument value in function signatures
    extern const std::string emptyStr;

    // Forward declaration of BlackBox response classes
    class Response;
    class BulkResponse;
    class SessionResp;
    class MultiSessionResp;
    class LoginResp;
    class HostResp;
    class PwdQualityResp;

    /// Helper class to make all accessors non-copyable
    class NonCopyable {
        NonCopyable(const NonCopyable& c);
        NonCopyable& operator=(const NonCopyable& c);
    public:
        NonCopyable() {};
    };

    /// Get regname from BB response
    struct Regname : public NonCopyable {
        Regname(const Response* pResp);

        operator const std::string&() const { return value(); }
        const std::string& value() const;

        std::string value_;
    };

    /// Get uid information from BB response
    struct Uid : public NonCopyable {
        Uid(const Response* pResp);
        ~Uid();

        const std::string& uid() const;
        bool hosted() const;

        std::string uid_;
        bool hosted_;

    private:
        struct Ext;                             // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    /// Get lite account information from BB response
    struct LiteUid : public NonCopyable {
        LiteUid(const Response* pResp);
        ~LiteUid();

        const std::string& liteUid() const;
        bool liteAccount() const;

        std::string liteUid_;
        bool liteAccount_;                      // not used yet, reserved for future

    private:
        struct Ext;                             // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    /// Get PDD user information from BB response
    struct PDDUserInfo : public NonCopyable {
        PDDUserInfo(const Response* pResp);
        ~PDDUserInfo();

        const std::string& domId() const;
        const std::string& domain() const;
        const std::string& mx() const;
        const std::string& domEna() const;
        const std::string& catchAll() const;

        std::string domId_, domain_, mx_;

    private:
        struct Ext;                             // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    /// Get karma information from BB response
    struct KarmaInfo : public NonCopyable {
        KarmaInfo(const Response* pResp);
        ~KarmaInfo();

        const std::string& karma() const;
        const std::string& banTime() const;
        const std::string& karmaStatus() const;

        std::string karma_, banTime_;

    private:
        struct Ext;                             // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    /// Get display name information from BB response
    struct DisplayNameInfo : public NonCopyable {
        DisplayNameInfo(const Response* pResp);
        ~DisplayNameInfo();

        const std::string& name() const;
        bool social() const;
        const std::string& socProfile() const;
        const std::string& socProvider() const;
        const std::string& socTarget() const;
        const std::string& defaultAvatar() const;

        std::string name_;

        bool social_;
        std::string socProfile_, socProvider_, socTarget_;

    private:
        struct Ext;                             // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    /// Get login information from BB response
    struct LoginInfo : public NonCopyable {
        LoginInfo(const Response* pResp);
        ~LoginInfo();

        const std::string& login() const;
        bool havePassword() const;
        bool haveHint() const;

        std::string login_;
        bool havePassword_;

    private:
        struct Ext;                             // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    /// Get email list from BB response
    class EmailList : public NonCopyable {
    public:
        /// Email list item
        struct Item {
            bool native() const;
            bool validated() const;
            bool isDefault() const;
            bool rpop() const;
            const std::string& address() const;
            const std::string& date() const;

            bool native_, validated_, default_, rpop_;
            std::string address_;
            std::string date_;

            Item();
            Item(const Item& i);
            ~Item();
            Item& operator= (const Item& i);
            bool operator== (const Item& i);

        private:
            struct Ext;                         // reserved for future extensions
            std::auto_ptr<Ext> pExt_;

        };
        typedef std::vector<Item> ListType;

        EmailList(const Response* pResp);

        bool empty() const { return list_.empty(); }
        size_t size() const { return list_.size(); }

        const std::string& getDefault() const { return pDefault_?pDefault_->address_:emptyStr; }
        const Item* getDefaultItem() const { return pDefault_; }
        const ListType& getEmailItems() const { return list_; }

    private:
        ListType list_;
        const Item* pDefault_;                  // pointer to default email, can be NULL
    };

    /// Get new session_id from BB sessionid response
    struct NewSessionId : public NonCopyable {
        NewSessionId(const SessionResp* pResp);
        NewSessionId(const MultiSessionResp* pResp);

        operator const std::string&() const { return value(); }

        const std::string& value() const;
        const std::string& domain() const;
        const std::string& expires() const;
        bool httpOnly() const;

        std::string value_;
        std::string domain_;
        std::string expires_;
        bool http_only_;
    };

    /// Get authorization info from BB sessionid response
    struct AuthInfo : public NonCopyable {
        AuthInfo(const SessionResp* pResp);
        ~AuthInfo();

        bool social() const;
        const std::string& age() const;
        const std::string& profileId() const;
        bool secure() const;

        bool social_;
        std::string age_, profileId_;
        bool secure_;

    private:
        struct Ext;                             // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    /// Get allow_more_users flag from BB sessionid response
    struct AllowMoreUsers : public NonCopyable {
        AllowMoreUsers(const MultiSessionResp* pResp);
        ~AllowMoreUsers();

        bool allowMoreUsers() const;

        bool allowMoreUsers_;

    private:
        struct Ext;                             // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    class AliasList : public NonCopyable {
    public:
        struct Item {
            enum Type {
                Login       = 1, // left out for compatibility, correct name: Portal
                Portal      = 1,
                Mail        = 2,
                NarodMail   = 3,
                Lite        = 5,
                Social      = 6,
                PDD         = 7,
                PDDAlias    = 8,
                AltDomain   = 9,
                Phonish     = 10,
                PhoneNumber = 11,
                Mailish     = 12,
                Yandexoid   = 13,
                KinopoiskId = 15,
                SSO         = 24,
            } type_;
            std::string alias_;

            Type type() const;
            const std::string& alias() const;

            Item();
            Item(const Item& i);
            ~Item();
            Item& operator= (const Item& i);
            bool operator== (const Item& i);

        private:
            struct Ext;                         // reserved for future extensions
            std::auto_ptr<Ext> pExt_;
        };
        typedef std::vector<Item> ListType;

        AliasList(const Response* pResp);

        bool empty() const { return list_.empty(); }
        size_t size() const { return list_.size(); }

        const ListType& getAliases() const { return list_; }

    private:
        ListType list_;
    };

    class MailHostsList : public NonCopyable {
    public:
        struct Item {

            const std::string& hostId() const;
            const std::string& hostIp() const;
            const std::string& hostName() const;
            const std::string& dbId() const;
            const std::string& sid() const;
            const std::string& priority() const;
            const std::string& mx() const;

            std::string hostId_, hostIp_, hostName_;
            std::string dbId_, sid_, priority_, mx_;

            Item();
            Item(const Item& i);
            ~Item();
            Item& operator= (const Item& i);
            bool operator== (const Item& i);

        private:
            struct Ext;                         // reserved for future extensions
            std::auto_ptr<Ext> pExt_;
        };

        typedef std::vector<Item> ListType;

        MailHostsList(const HostResp* pResp);

        bool empty() const { return list_.empty(); }
        size_t size() const { return list_.size(); }

        const ListType& getHosts() const { return list_; }

    private:
        ListType list_;
    };

    class OAuthInfo : public NonCopyable {
    public:
        typedef std::map< std::string, std::string > MapType;

        OAuthInfo(const SessionResp* pResp);

        bool empty() const { return info_.empty(); }
        size_t size() const { return info_.size(); }

        const MapType& getInfo() const { return info_; }

    private:
        MapType info_;
    };

    struct BruteforcePolicy : public NonCopyable {
        enum Mode {
            None,
            Captcha,
            Delay,
            Expired
        } mode_;

        BruteforcePolicy(const LoginResp* pResp);
        ~BruteforcePolicy();

        int delay() const;
        Mode mode() const;

        int delay_;

    private:
        struct Ext;                         // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    struct SessionKind : public NonCopyable {
        enum Kind {
            None = 0,
            Support = 1,
            Stress = 2
        } kind_;

        SessionKind(const SessionResp* pResp);
        SessionKind(const MultiSessionResp* pResp);

        Kind kind() const;
        const std::string& kindName() const;

        std::string kindName_;
    };

    struct AuthId : public NonCopyable {
        AuthId(const SessionResp* pResp);
        AuthId(const MultiSessionResp* pResp);
        ~AuthId();

        operator const std::string&() const { return authId(); }

        const std::string& time() const;
        const std::string& userIp() const;
        const std::string& hostId() const;
        const std::string& authId() const;

        std::string time_, userIp_, hostId_, authId_;

    private:
        struct Ext;                             // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    struct PasswordQuality : public NonCopyable {

        PasswordQuality(const LoginResp* pResp);
        PasswordQuality(const PwdQualityResp* pResp);
        ~PasswordQuality();

        operator const std::string&() const { return quality(); }
        const std::string& quality() const;

        std::string quality_;

    private:
        struct Ext;                         // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    struct ConnectionId : public NonCopyable {

        ConnectionId(const LoginResp* pResp);
        ConnectionId(const SessionResp* pResp);
        ConnectionId(const MultiSessionResp* pResp);
        ~ConnectionId();

        operator const std::string&() const { return connectionId(); }
        const std::string& connectionId() const;

        std::string connectionId_;

    private:
        struct Ext;                         // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    struct LoginId : public NonCopyable {

        LoginId(const LoginResp* pResp);
        LoginId(const SessionResp* pResp);
        LoginId(const MultiSessionResp* pResp);
        ~LoginId();

        operator const std::string&() const { return loginId(); }
        const std::string& loginId() const;

        std::string loginId_;

    private:
        struct Ext;                         // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

    struct UserTicket : public NonCopyable {
        UserTicket(const Response* resp);
        ~UserTicket();

        const std::string& get() const {
            return userTicket_;
        }

    private:
        std::string userTicket_;

        struct Ext;                         // reserved for future extensions
        std::auto_ptr<Ext> pExt_;
    };

}   // namespace bb

#endif  // BLACKBOX_ACCESSORS_H_INCLUDED

// vi: expandtab:sw=4:ts=4
// kate: replace-tabs on; indent-width 4; tab-width 4;
