#pragma once
//  blackbox.h
//
// Copyright (c) Yandex, LLC, 2010. All rights reserved.
//
// Provides C++ binding for the BlackBox HTTP/XML API

#ifndef BLACKBOX_BLACKBOX2_H_INCLUDED
#define BLACKBOX_BLACKBOX2_H_INCLUDED

#include <stdexcept>
#include <string>
#include <vector>
#include <memory>

#include <library/cpp/blackbox2/session_errors.h>

// options.h contains blackbox request option classes and common options
#include <yandex/blackbox/options.h>

// accessors.h contains blackbox response information accessors
#include <yandex/blackbox/accessors.h>

namespace bb {
    using namespace NBlackbox2;

    /// Default argument value in function signatures
    extern const std::string emptyStr;

    // ==== Response structure ============================================

    /// Basic response type
    class Response {
    public:
        class Impl;

        Response(std::auto_ptr<Impl> pImpl);
        virtual ~Response();

        const std::string& message() const { return message_; }

        Impl* getImpl() const { return pImpl_.get(); }    // backdoor for info accessors

    private:
        Response(const Response& r);               // not allowed to copy
        Response& operator= (const Response& r);   // not allowed to copy

    protected:
        std::auto_ptr<Impl> pImpl_;
        std::string message_;
    };

    /// Bulk userinfo response
    class BulkResponse: public Response {
    public:
        BulkResponse(std::auto_ptr<Impl> pImpl);
        ~BulkResponse();

        int count() const { return count_; }

        std::string id(int i) const;
        std::auto_ptr<Response> user(int i) const;

    protected:
        int count_;
    };

    // ==== Blackbox methods ==============================================

    /// Options for signing request
    class ConsumerContext {
    public:
        ConsumerContext(const std::string& consumer, const std::string& clientId, const std::string& base64Secret);

        const std::string& consumer() const {
            return consumer_;
        }
        const std::string& clientId() const {
            return clientId_;
        }
        const std::string& secret() const {
            return secret_;
        }

    private:
        std::string consumer_;
        std::string clientId_;
        std::string secret_;
    };

    /// User identification by login or login/sid
    struct LoginSid {
        std::string login;
        std::string sid;

        LoginSid(const std::string& login, const std::string& sid = emptyStr)     // sid can be empty
            : login(login), sid(sid) {}
    };

    /// User identification by suid/sid
    struct SuidSid {
        std::string suid;
        std::string sid;

        SuidSid(const std::string& suid, const std::string& sid)
            : suid(suid), sid(sid) {}
    };

    // method = userinfo

    std::string InfoRequest(
        const std::string& uid,
        const std::string& userIp,
        const Options& options = optNone);
    std::string InfoRequestSigned(
        const std::string& uid,
        const std::string& userIp,
        const ConsumerContext& signCtx,
        const Options& options = optNone,
        const std::string& meta = emptyStr);

    std::string InfoRequest(
        const LoginSid& loginsid,
        const std::string& userIp,
        const Options& options = optNone);
    std::string InfoRequestSigned(
        const LoginSid& loginsid,
        const std::string& userIp,
        const ConsumerContext& signCtx,
        const Options& options = optNone,
        const std::string& meta = emptyStr);

    std::string InfoRequest(
        const SuidSid& suidsid,
        const std::string& userIp,
        const Options& options = optNone);
    std::string InfoRequestSigned(
        const SuidSid& suidsid,
        const std::string& userIp,
        const ConsumerContext& signCtx,
        const Options& options = optNone,
        const std::string& meta = emptyStr);

    // Bulk userinfo request
    std::string InfoRequest(
        const std::vector<std::string>& uids,
        const std::string& userIp,
        const Options& options = optNone);
    std::string InfoRequestSigned(
        const std::vector<std::string>& uids,
        const std::string& userIp,
        const ConsumerContext& signCtx,
        const Options& options = optNone,
        const std::string& meta = emptyStr);

    /// Parse given BB userinfo reply text and construct Response object
    std::auto_ptr<Response> InfoResponse (const std::string& bbResponse);

    /// Parse given BB bulk userinfo reply text and construct BulkResponse object
    std::auto_ptr<BulkResponse> InfoResponseBulk (const std::string& bbResponse);

    // method = login

    /// Get request URI and POST data for login request
    struct LoginReqData {
        std::string uri_;
        std::string postData_;
    };

    LoginReqData LoginRequest(
        const LoginSid& loginsid,
        const std::string& password,
        const std::string& authtype,
        const std::string& userIp,
        const Options& options = optNone );
    LoginReqData LoginRequestSigned(
        const LoginSid& loginsid,
        const std::string& password,
        const std::string& authtype,
        const std::string& userIp,
        const ConsumerContext& signCtx,
        const Options& options = optNone,
        const std::string& meta = emptyStr );

    LoginReqData LoginRequestUid(
        const std::string& uid,
        const std::string& password,
        const std::string& authtype,
        const std::string& userIp,
        const Options& options = optNone );
    LoginReqData LoginRequestUidSigned(
        const std::string& uid,
        const std::string& password,
        const std::string& authtype,
        const std::string& userIp,
        const ConsumerContext& signCtx,
        const Options& options = optNone,
        const std::string& meta = emptyStr );

    /// Blackbox login method response
    class LoginResp: public Response {
    public:
        // Login v1 status
        enum Status {
            Valid,
            Disabled,
            Invalid,
            ShowCaptcha,
            AlienDomain,
            Compromised,
            Expired
        };

        // Login v2 detailed status: Account+Password
        enum AccountStatus {
            accUnknown,
            accValid,
            accAlienDomain,
            accNotFound,
            accDisabled
        };

        enum PasswdStatus {
            pwdUnknown,
            pwdValid,
            pwdBad,
            pwdCompromised
        };

        LoginResp(std::auto_ptr<Impl> pImpl);
        ~LoginResp();

        Status status() const { return status_; }
        AccountStatus accStatus() const { return accStatus_; }
        PasswdStatus pwdStatus() const { return pwdStatus_; }

    private:
        Status status_;
        AccountStatus accStatus_;
        PasswdStatus pwdStatus_;
    };

    /// Parse given BB login reply text and construct LoginResp object
    std::auto_ptr<LoginResp> LoginResponse (const std::string& bbResponse);

    // method = sessionid

    std::string SessionIDRequest(
        const std::string& sessionId,
        const std::string& hostname,
        const std::string& userIp,
        const Options& options = optNone );
    std::string SessionIDRequestSigned(
        const std::string& sessionId,
        const std::string& hostname,
        const std::string& userIp,
        const ConsumerContext& signCtx,
        const Options& options = optNone,
        const std::string& meta = emptyStr );

    // method = oauth

    LoginReqData OAuthRequestSecure(
        const std::string& token,
        const std::string& userIp,
        const Options& options = optNone );
    LoginReqData OAuthRequestSignedSecure(
        const std::string& token,
        const std::string& userIp,
        const ConsumerContext& signCtx,
        const Options& options = optNone,
        const std::string& meta = emptyStr );

    /// Blackbox sessionid method response
    class SessionResp: public Response {
    public:
        enum Status {
            Valid,
            NeedReset,
            Expired,
            NoAuth,
            Disabled,
            Invalid,
            WrongGuard = 8,
        };
        SessionResp(std::auto_ptr<Impl> pImpl);
        ~SessionResp();

        Status status() const { return status_; }
        const std::string& age() const { return age_; }
        bool isLite() const { return isLite_; }

    private:
        Status status_;
        std::string age_;
        bool isLite_;
    };

    /// Blackbox sessionid method response, multisession mode
    class MultiSessionResp: public Response {
    public:
        MultiSessionResp(std::auto_ptr<Impl> pImpl);
        ~MultiSessionResp();

        SessionResp::Status status() const { return status_; }
        const std::string& age() const { return age_; }
        const std::string& defaultUid() const { return default_uid_; }

        int count() const { return count_; }

        std::string id(int i) const;
        std::auto_ptr<SessionResp> user(int i) const;

    private:
        SessionResp::Status status_;
        std::string age_;
        std::string default_uid_;
        int count_;
    };

    /// Parse given BB sessionid reply text and construct SessionResp object
    std::auto_ptr<SessionResp> SessionIDResponse (const std::string& bbResponse);
    std::auto_ptr<MultiSessionResp> SessionIDResponseMulti (const std::string& bbResponse);

    // method = mailhost

    std::string MailHostCreateRequest(
        const std::string& scope,
        const std::string& dbId,
        const std::string& priority,
        const std::string& mx);

    std::string MailHostDeleteRequest(
        const std::string& scope,
        const std::string& dbId);

    std::string MailHostAssignRequest(
        const std::string& scope,
        const std::string& suid,
        const std::string& dbId,
        const std::string& oldDbId = emptyStr);

    std::string MailHostSetPriorityRequest(
        const std::string& scope,
        const std::string& dbId,
        const std::string& priority);

    std::string MailHostFindRequest(
        const std::string& scope,
        const std::string& priority);

    class HostResp {
    public:
        enum Status {
            OK,
            AccessDenied,
            MissingParam,
            UnknownOp,
            UnknownScope,
            InvalidParam,
            RecordNotFound
        };
        HostResp(std::auto_ptr<Response::Impl> pImpl);
        ~HostResp();

        Status status() const { return status_; }
        const std::string& message() const { return message_; }

        Response::Impl* getImpl() const { return pImpl_.get(); }    // backdoor for info accessors

    protected:
        std::auto_ptr<Response::Impl> pImpl_;

    private:
        Status status_;
        std::string message_;
    };

    /// Parse given BB mailhost reply text and construct HostResp object
    std::auto_ptr<HostResp> MailHostResponse (const std::string& bbResponse);

    // method = pwdquality

    std::string PwdQualityRequest(
        const std::string& sessionId,
        const std::string& hostname);

    class PwdQualityResp {
    public:
        PwdQualityResp(std::auto_ptr<Response::Impl> pImpl);
        ~PwdQualityResp();

        const std::string& message() const { return message_; }

        Response::Impl* getImpl() const { return pImpl_.get(); }    // backdoor for info accessors

    protected:
        std::auto_ptr<Response::Impl> pImpl_;

    private:
        std::string message_;
    };

    /// Parse given BB password quality reply text and construct PwdQualityResp object
    std::auto_ptr<PwdQualityResp> PwdQualityResponse (const std::string& bbResponse);

    // ==== Exceptions ====================================================

    /// Generic Blackbox exception
    class BBError: public std::runtime_error {
        NSessionCodes::ESessionError code_;

    public:
        BBError(NSessionCodes::ESessionError code, const std::string& message)
            : std::runtime_error (message), code_(code) {}
        virtual ~BBError() {}

        NSessionCodes::ESessionError code() const { return code_; }
    };

    /// Fatal errors: in case of such an error repeating request doesn't
    /// make sense (e.g. bad argument)
    class FatalError: public BBError {
    public:
        FatalError(NSessionCodes::ESessionError code, const std::string& message) :
            BBError(code, message) {}
    };

    /// Temporary error: in case of such an error it makes sense to repeat
    /// request (e.g. network ot database timeout)
    class TempError: public BBError {
    public:
        TempError(NSessionCodes::ESessionError code, const std::string& message) :
            BBError(code, message) {}
    };

}   // namespace bb

#endif /* BLACKBOX_BLACKBOX2_H_INCLUDED */

// vi: expandtab:sw=4:ts=4
// kate: replace-tabs on; indent-width 4; tab-width 4;
