//  options.h
//
// Copyright (c) Yandex, LLC, 2010. All rights reserved.
//
// Provides request options for the BlackBox HTTP/XML API

#ifndef BLACKBOX_OPTIONS_H_INCLUDED
#define BLACKBOX_OPTIONS_H_INCLUDED

#include <string>
#include <vector>
#include <map>
#include <set>

namespace bb {

    /// Default argument value in function signatures
    extern const std::string emptyStr;

    /// Single blackbox request option
    class Option {
    public:
        Option (const std::string& key, const std::string& val)
            : key_(key), val_(val) {}

        const std::string& key() const { return key_; }
        const std::string& val() const { return val_; }

    private:
        std::string key_, val_;
    };

    // Standard blackbox options

    extern const Option optRegname;                 // show name as used for registration
    extern const Option optAuthId;                  // return authid information
    extern const Option optFullInfo;                // return full user info in login and sessionid

    extern const Option optVersion2;                // use login version 2 mode
    extern const Option optMultisession;            // use multisession response format

    extern const Option optGetSocialAliases;        // get social aliases info
    extern const Option optGetAllAliases;           // get all aliases info

    extern const Option optGetAllEmails;            // get all known user emails
    extern const Option optGetYandexEmails;         // get all user Yandex emails
    extern const Option optGetDefaultEmail;         // get default email

    extern const Option optGetUserTicket;           // get user ticket

    /// Blackbox request options list
    class Options {
    public:
        Options () {}
        Options (const Option& opt);

        typedef std::vector<Option> ListType;

        /// Return options list
        const ListType& list() const { return options_; }

        /// Add option to the list
        Options& operator<< (const Option& opt);

    private:
        ListType options_;
    };

    /// Default value for function signatures, no options
    extern const Options optNone;

    // Forward declaration of BlackBox response class
    class Response;

    /// DB_fields options, inout parameter of blackbox methods
    class DBFields {
    public:
        typedef std::map<std::string, std::string>  container_type;
        typedef container_type::iterator            iterator;
        typedef container_type::const_iterator      const_iterator;

        DBFields (const std::string& field = emptyStr); // add one field at construction for convenience
        DBFields (const Response* pResp);               // init fields and values by response

        void readResponse (const Response* pResp);      // parse the response and get values

        bool empty() const  { return fields_.empty(); } // true if no fields inside
        size_t size() const { return fields_.size(); }  // number of fields inside

        void clear();                                   // clear all stored dbfields
        void clearValues();                             // clear stored values but keep field names

        DBFields& operator<< (const std::string& field);// add field

        const std::string& get (const std::string& field) const; // get field value by name
        bool has(const std::string& field) const;       // has field

        const_iterator begin() const;                   // iterate over fields
        const_iterator end() const;                     // needed for *-fields

        operator Options() const;                       // to pass DBFields as Options

    private:
        container_type fields_;
    };

    Options& operator<< (Options& options, const DBFields& fields);

    /// Test email request option
    class OptTestEmail {
    public:
        OptTestEmail (const std::string& addr) : addrToTest_(addr) {}
        Options& format (Options& opts) const;

    private:
        std::string addrToTest_;
    };

    Options& operator<< (Options& options, const OptTestEmail& opt);

    /// Aliases request option
    class OptAliases {
    public:
        OptAliases (const std::string& alias = emptyStr);   // add one alias at construction for convenience

        bool empty() const  { return aliases_.empty(); }    // true if no aliases inside
        size_t size() const { return aliases_.size(); }     // number of aliases inside

        OptAliases& operator<< (const std::string& alias);  // add alias

        operator Options() const;                           // to pass aliases as Options
        Option format () const;                             // format request option
    private:
        std::set<std::string> aliases_;
    };

    Options& operator<< (Options& options, const OptAliases& opt);

    /// Attributes request option and accessor
    class Attributes {
    public:
        typedef std::map<std::string, std::string>  container_type;
        typedef container_type::iterator            iterator;
        typedef container_type::const_iterator      const_iterator;

        Attributes (const std::string& attr = emptyStr);    // add one attr at construction for convenience
        Attributes (const Response* pResp);                 // init fields and values by response

        void readResponse (const Response* pResp);          // parse the response and get values

        bool empty() const  { return attributes_.empty(); } // true if no fields inside
        size_t size() const { return attributes_.size(); }  // number of fields inside

        void clear();                                       // clear all stored attributes
        void clearValues();                                 // clear stored values but keep field names

        Attributes& operator<< (const std::string& attr);   // add attribute

        const std::string& get (const std::string& attr) const; // get attribute value by name
        bool has(const std::string& attr) const;            // has attribute

        const_iterator begin() const;                       // container iterators
        const_iterator end() const;                         // container iterators

        operator Options() const;                           // to pass Attributes as Options
        Option format () const;                             // format request option

    private:
        container_type attributes_;
    };

    Options& operator<< (Options& options, const Attributes& attrs);

}   // namespace bb


#endif  // BLACKBOX_OPTIONS_H_INCLUDED

// vi: expandtab:sw=4:ts=4
// kate: replace-tabs on; indent-width 4; tab-width 4;
