#include <boost/python/def.hpp>
#include <boost/python/enum.hpp>
#include <boost/python/copy_const_reference.hpp>
#include <boost/python/suite/indexing/vector_indexing_suite.hpp>
#include <boost/python/exception_translator.hpp>
#include <boost/python/overloads.hpp>
#include <boost/python/module.hpp>


#include "yandex/blackbox/blackbox2.h"
#include "python_response.h"

namespace py = boost::python;

namespace bb {

    void register_options();
    void register_accessors();
    void register_responses();

    //LoginReqData (*LoginRequestLogin) (const LoginSid&, const std::string&, const std::string&, const std::string&, const Options&) = LoginRequest;
    //LoginReqData (*LoginRequestUid) (const std::string&, const std::string&, const std::string&, const std::string&, const Options&) = LoginRequest;

    BOOST_PYTHON_FUNCTION_OVERLOADS(MailHostAssign_overloads, MailHostAssignRequest, 3, 4);
    BOOST_PYTHON_FUNCTION_OVERLOADS(SessionID_overloads, SessionIDRequest, 3, 4);
    BOOST_PYTHON_FUNCTION_OVERLOADS(OAuth_overloads, OAuthRequest, 2, 3);
    BOOST_PYTHON_FUNCTION_OVERLOADS(LoginRequest_overloads, LoginRequest, 4, 5);
    BOOST_PYTHON_FUNCTION_OVERLOADS(LoginRequestUid_overloads, LoginRequestUid, 4, 5);
    BOOST_PYTHON_FUNCTION_OVERLOADS(InfoRequest_overloads, InfoRequest, 2, 3);

    static void register_requests() {
        // class LoginSid
        py::class_<LoginSid> loginSid("LoginSid", "Login/Sid parameter pair",
                                      py::init<std::string, std::string>() );
        loginSid.def(py::init<std::string>() );
        loginSid.def_readwrite("login", &LoginSid::login);
        loginSid.def_readwrite("sid", &LoginSid::sid);

        // class SuidSid
        py::class_<SuidSid> suidSid("SuidSid", "Suid/Sid parameter pair",
                                      py::init<std::string, std::string>() );
        suidSid.def_readwrite("suid", &SuidSid::suid);
        suidSid.def_readwrite("sid", &SuidSid::sid);

        // InfoRequest and overloads
        std::string (*InfoRequest_uid) (const std::string&,
            const std::string&, const Options&) = InfoRequest;

        std::string (*InfoRequest_lsid) (const LoginSid&,
            const std::string&, const Options&) = InfoRequest;

        std::string (*InfoRequest_ssid) (const SuidSid&,
            const std::string&, const Options&) = InfoRequest;

        std::string (*InfoRequest_uids) (const std::vector<std::string>&,
            const std::string&, const Options&) = InfoRequest;

        py::class_<std::vector<std::string> >("UidsVector")
            .def(py::vector_indexing_suite<std::vector<std::string> >() );

        py::def("InfoRequest", InfoRequest_uid, InfoRequest_overloads());
        py::def("InfoRequest", InfoRequest_lsid, InfoRequest_overloads());
        py::def("InfoRequest", InfoRequest_ssid, InfoRequest_overloads());
        py::def("InfoRequest", InfoRequest_uids, InfoRequest_overloads());

        // LoginRequest method
        py::class_<LoginReqData> lrecd("LoginReqData", "Login request data: GET url and POST body");
        lrecd.def_readwrite("uri", &LoginReqData::uri_);
        lrecd.def_readwrite("postdata", &LoginReqData::postData_);

        py::def("LoginRequest", &LoginRequest, LoginRequest_overloads());
        py::def("LoginRequest", &LoginRequestUid, LoginRequestUid_overloads());

        // SessionIDRequest method
        py::def("SessionIDRequest", &SessionIDRequest, SessionID_overloads());

        // OAuthRequest method
        py::def("OAuthRequest", &OAuthRequest, OAuth_overloads());

        // MailHost methods
        py::def("MailHostCreateRequest", &MailHostCreateRequest);
        py::def("MailHostDeleteRequest", &MailHostDeleteRequest);
        py::def("MailHostAssignRequest", &MailHostAssignRequest, MailHostAssign_overloads());
        py::def("MailHostSetPriorityRequest", &MailHostSetPriorityRequest);
        py::def("MailHostFindRequest", &MailHostFindRequest);

        // PwdQualityRequest method
        py::def("PwdQualityRequest", &PwdQualityRequest);
    }

    // Helpers for throwing custom exceptions to python
    PyObject *bbErrType = NULL;
    PyObject *fatalErrType = NULL;
    PyObject *tempErrType = NULL;

    static void bb_err_handler(const BBError& err) {
       py::object exception_instance(err);
       PyErr_SetObject(bbErrType, exception_instance.ptr());
    }

    static void fatal_err_handler(const FatalError& err) {
       py::object exception_instance(err);
       PyErr_SetObject(fatalErrType, exception_instance.ptr());
    }

    static void temp_err_handler(const TempError& err) {
       py::object exception_instance(err);
       PyErr_SetObject(tempErrType, exception_instance.ptr());
    }


    static void register_exceptions() {
        py::enum_<NSessionCodes::ESessionError> errcodes("SessionError");
        errcodes.value("OK",                NSessionCodes::OK);
        errcodes.value("Unknown",           NSessionCodes::UNKNOWN);
        errcodes.value("InvalidParams",     NSessionCodes::INVALID_PARAMS);
        errcodes.value("DBFetchFailed",     NSessionCodes::DB_FETCHFAILED);
        errcodes.value("DBException",       NSessionCodes::DB_EXCEPTION);
        errcodes.value("AccessDenied",      NSessionCodes::ACCESS_DENIED);

        py::class_<BBError> err("BBError", "Generic blackbox library exception", py::no_init);
        err.def(py::init<NSessionCodes::ESessionError, std::string>());
        err.def("code", &BBError::code);
        err.def("what", &BBError::what);
        bbErrType = err.ptr();

        py::class_<FatalError, py::bases<BBError> > fatalerr("FatalError", "Fatal blackbox library error", py::no_init);
        fatalerr.def(py::init<NSessionCodes::ESessionError, std::string>());
        fatalErrType = fatalerr.ptr();

        py::class_<TempError, py::bases<BBError> > tmperr("TempError", "Temporary blackbox error, retry possible", py::no_init);
        tmperr.def(py::init<NSessionCodes::ESessionError, std::string>());
        tempErrType = tmperr.ptr();

        py::register_exception_translator<BBError>(bb_err_handler);
        py::register_exception_translator<FatalError>(fatal_err_handler);
        py::register_exception_translator<TempError>(temp_err_handler);
    }

    BOOST_PYTHON_MODULE(blackbox2) {
        register_options();
        register_requests();
        register_responses();
        register_accessors();
        register_exceptions();
    }

} // namespace bb

// vi: expandtab:sw=4:ts=4
// kate: replace-tabs on; indent-width 4; tab-width 4;
